#!/usr/bin/env python3
"""Verification script for roformer-separation package."""

import sys

def verify_imports():
    """Verify all package imports work."""
    print("✓ Testing imports...")
    try:
        from roformer_separation import MelBandRoformer
        from roformer_separation.download import get_model_paths, get_cache_dir
        from roformer_separation.config import load_config_from_yaml, build_default_config
        from roformer_separation.inference import demix, normalize_audio
        from roformer_separation.model import Attention, Transformer
        from roformer_separation.cli import main
        print("  ✓ All imports successful")
        return True
    except ImportError as e:
        print(f"  ✗ Import failed: {e}")
        return False


def verify_version():
    """Verify package version."""
    print("✓ Testing package version...")
    try:
        import roformer_separation
        version = roformer_separation.__version__
        print(f"  ✓ Package version: {version}")
        return True
    except Exception as e:
        print(f"  ✗ Version check failed: {e}")
        return False


def verify_cache_dir():
    """Verify cache directory creation."""
    print("✓ Testing cache directory...")
    try:
        from roformer_separation.download import get_cache_dir
        cache_dir = get_cache_dir()
        if cache_dir.exists():
            print(f"  ✓ Cache directory: {cache_dir}")
            return True
        else:
            print(f"  ✗ Cache directory not created: {cache_dir}")
            return False
    except Exception as e:
        print(f"  ✗ Cache directory test failed: {e}")
        return False


def verify_model_config():
    """Verify model configuration."""
    print("✓ Testing model configuration...")
    try:
        from roformer_separation.config import build_default_config
        config = build_default_config()
        
        # Check required config sections
        assert hasattr(config, 'audio')
        assert hasattr(config, 'model')
        assert hasattr(config, 'training')
        assert hasattr(config, 'inference')
        
        # Check key parameters
        assert config.audio.sample_rate == 44100
        assert config.model.dim == 384
        assert config.model.num_stems == 2
        
        print("  ✓ Default config is valid")
        return True
    except Exception as e:
        print(f"  ✗ Config test failed: {e}")
        return False


def verify_model_init():
    """Verify model initialization."""
    print("✓ Testing model initialization...")
    try:
        from roformer_separation import MelBandRoformer
        from roformer_separation.config import build_default_config
        
        config = build_default_config()
        model = MelBandRoformer(**vars(config.model))
        
        param_count = sum(p.numel() for p in model.parameters())
        print(f"  ✓ Model initialized with {param_count:,} parameters")
        return True
    except Exception as e:
        print(f"  ✗ Model init failed: {e}")
        return False


def verify_available_instruments():
    """Verify available instruments."""
    print("✓ Testing available instruments...")
    try:
        from roformer_separation.download import MODEL_CONFIGS
        instruments = list(MODEL_CONFIGS.keys())
        print(f"  ✓ Available instruments: {', '.join(instruments)}")
        return True
    except Exception as e:
        print(f"  ✗ Instrument check failed: {e}")
        return False


def verify_yaml_loading():
    """Verify YAML config loading with python/tuple tags."""
    print("✓ Testing YAML config loading...")
    try:
        from roformer_separation.download import get_model_paths
        from roformer_separation.config import load_config_from_yaml
        
        # This will download if not cached
        _, config_path = get_model_paths('vocals')
        config = load_config_from_yaml(str(config_path))
        
        # Verify the tuple was loaded correctly
        window_sizes = config.model.multi_stft_resolutions_window_sizes
        assert isinstance(window_sizes, tuple), f"Expected tuple, got {type(window_sizes)}"
        assert len(window_sizes) == 5, f"Expected 5 window sizes, got {len(window_sizes)}"
        
        print(f"  ✓ YAML loaded with python/tuple: {window_sizes}")
        return True
    except Exception as e:
        print(f"  ✗ YAML loading failed: {e}")
        return False


def main():
    """Run all verification tests."""
    print("=" * 60)
    print("Roformer-Separation Package Verification")
    print("=" * 60)
    print()
    
    tests = [
        verify_imports,
        verify_version,
        verify_cache_dir,
        verify_model_config,
        verify_model_init,
        verify_available_instruments,
        verify_yaml_loading,
    ]
    
    results = []
    for test in tests:
        try:
            result = test()
            results.append(result)
        except Exception as e:
            print(f"  ✗ Test crashed: {e}")
            results.append(False)
        print()
    
    print("=" * 60)
    passed = sum(results)
    total = len(results)
    
    if passed == total:
        print(f"✓ All {total} tests passed!")
        print("=" * 60)
        print()
        print("Package is ready to use!")
        print()
        print("Try the CLI:")
        print("  roformer-separate --help")
        print()
        return 0
    else:
        print(f"✗ {total - passed} of {total} tests failed")
        print("=" * 60)
        return 1


if __name__ == "__main__":
    sys.exit(main())

