# Installation and Testing Guide

## Package Structure

The `roformer-separation` package is now a fully pip-installable Python package with the following structure:

```
roformer-separation/
├── roformer_separation/          # Main package directory
│   ├── __init__.py              # Package initialization
│   ├── __main__.py              # Entry point for python -m
│   ├── cli.py                   # Command-line interface
│   ├── config.py                # Configuration management
│   ├── download.py              # Model downloader
│   ├── inference.py             # Inference utilities
│   └── model.py                 # MelBandRoformer model
├── pyproject.toml               # Package configuration
├── setup.py                     # Backwards compatibility
├── README.md                    # User documentation
├── LICENSE                      # MIT License
├── MANIFEST.in                  # Package manifest
└── .gitignore                   # Git ignore rules
```

## Installation

### Option 1: Development Installation (Editable)

For development and testing:

```bash
cd /mnt/sc-dev/roformer-separation
pip install -e .
```

This creates a "editable" install where changes to the source code are immediately reflected.

### Option 2: Regular Installation (Local)

For regular use from local source:

```bash
pip install /mnt/sc-dev/roformer-separation
```

### Option 3: From PyPI (Future)

Once published to PyPI:

```bash
pip install roformer-separation
```

## Verification

### 1. Verify CLI Command

```bash
roformer-separate --help
```

Expected output: Help message with all options

### 2. Verify Python Import

```bash
python -c "from roformer_separation import MelBandRoformer; print('Success!')"
```

### 3. Verify Module Execution

```bash
python -m roformer_separation --help
```

### 4. Check Installed Version

```bash
python -c "import roformer_separation; print(roformer_separation.__version__)"
```

Expected output: `0.1.0`

## Quick Test

### Test with Sample Command (without actual audio)

```bash
# This will show error but verifies the CLI works
roformer-separate --instrument vocals --input_folder ./nonexistent --store_dir ./output
```

### Test Model Download (Optional)

To verify the download functionality works:

```bash
python -c "from roformer_separation.download import get_model_paths; paths = get_model_paths('vocals'); print(f'Downloaded to: {paths}')"
```

This will download the vocals model and config to `~/.cache/roformer-separation/`

## Usage Examples

### Basic Usage

```bash
# Separate vocals from an audio file
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/song.wav \
    --store_dir ./output

# Separate guitar from a folder of files
roformer-separate \
    --instrument guitar \
    --input_folder /path/to/audio_folder \
    --store_dir ./output
```

### Advanced Usage

```bash
# With instrumental extraction
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/audio \
    --store_dir ./output \
    --extract_instrumental

# With test-time augmentation (slower but better quality)
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/audio \
    --store_dir ./output \
    --use_tta

# Force CPU usage
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/audio \
    --store_dir ./output \
    --force_cpu

# Output as FLAC
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/audio \
    --store_dir ./output \
    --output_format flac
```

### Using Custom Models (Advanced)

```bash
roformer-separate \
    --instrument vocals \
    --input_folder /path/to/audio \
    --store_dir ./output \
    --checkpoint /path/to/custom_model.ckpt \
    --config /path/to/custom_config.yaml
```

## Python API Usage

```python
from roformer_separation import MelBandRoformer
from roformer_separation.download import get_model_paths
from roformer_separation.config import load_config_from_yaml
from roformer_separation.inference import demix
import librosa
import torch

# Download and get model paths
checkpoint_path, config_path = get_model_paths("vocals")

# Load config
config = load_config_from_yaml(str(config_path))

# Initialize model
model = MelBandRoformer(**vars(config.model))

# Load checkpoint
state_dict = torch.load(checkpoint_path, map_location='cpu')
if 'state' in state_dict:
    state_dict = state_dict['state']
model.load_state_dict(state_dict, strict=False)

# Setup device and move model
device = "cuda" if torch.cuda.is_available() else "cpu"
model = model.to(device)
model.eval()

# Load audio
audio, sr = librosa.load("song.wav", sr=44100, mono=False)

# Perform separation
separated = demix(config, model, audio, device, model_type='mel_band_roformer')

# Access results
vocals = separated['Vocals']
instrumental = separated['Instrumental']
```

## Uninstallation

```bash
pip uninstall roformer-separation
```

## Troubleshooting

### Import Errors

If you get import errors, ensure the package is installed:
```bash
pip list | grep roformer
```

### Command Not Found

If `roformer-separate` is not found:
1. Ensure the package is installed: `pip show roformer-separation`
2. Check your PATH includes the pip binary directory
3. Try using: `python -m roformer_separation` instead

### Model Download Issues

If downloads fail:
1. Check internet connectivity
2. Try manual download from HuggingFace URLs
3. Use `--checkpoint` and `--config` to specify local paths

### GPU/CUDA Issues

If you have CUDA issues:
1. Use `--force_cpu` flag
2. Verify PyTorch CUDA installation: `python -c "import torch; print(torch.cuda.is_available())"`
3. Update PyTorch to match your CUDA version

## Building Distribution Package

To create a distribution package for upload to PyPI:

```bash
# Install build tools
pip install build twine

# Build the package
python -m build

# This creates:
# - dist/roformer_separation-0.1.0.tar.gz (source distribution)
# - dist/roformer_separation-0.1.0-py3-none-any.whl (wheel)
```

## Publishing to PyPI (For Maintainers)

```bash
# Test PyPI first
twine upload --repository testpypi dist/*

# Production PyPI
twine upload dist/*
```

## Development Setup

For development:

```bash
# Clone repository
git clone https://github.com/xavriley/roformer-separation.git
cd roformer-separation

# Create virtual environment
python -m venv venv
source venv/bin/activate  # or `venv\Scripts\activate` on Windows

# Install in editable mode with dev dependencies
pip install -e ".[dev]"

# Run tests (when available)
pytest

# Format code
black roformer_separation/

# Check linting
flake8 roformer_separation/
```

## System Requirements

- Python 3.8 or higher
- 4GB+ RAM (8GB+ recommended)
- GPU with CUDA support (optional but recommended)
- Internet connection (for first-time model download)

## Dependencies

Core dependencies are automatically installed:
- torch>=2.0.0
- numpy
- librosa
- soundfile
- einops
- rotary-embedding-torch
- tqdm
- pyyaml

## Support

For issues, questions, or contributions:
- GitHub Issues: https://github.com/xavriley/roformer-separation/issues
- Documentation: See README.md

## License

MIT License - See LICENSE file for full text.

