//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Entrypoint and type definitions for Live Share for VS Code extension API
 */
const vscode = require("vscode");
const liveShareApiVersion = require('./package.json').version;
/**
 * Extension ID of the Live Share extension for VS Code.
 */
exports.extensionId = 'ms-vsliveshare.vsliveshare';
/**
 * Entrypoint for access to the Live Share API.
 *
 * @returns an instance of the Live Share API, or `null` if the Live Share extension
 * is not installed or failed to activate.
 *
 * @example To access the Live Share API from another extension:
 *
 *     import * as vsls from 'vsls/vscode';
 *     const liveshare = await vsls.getApi();
 */
async function getApi(callingExtensionId) {
    const liveshareExtension = vscode.extensions.getExtension(exports.extensionId);
    if (!liveshareExtension) {
        // The extension is not installed.
        return null;
    }
    const extensionApi = liveshareExtension.isActive ?
        liveshareExtension.exports : await liveshareExtension.activate();
    if (!extensionApi) {
        // The extensibility API is not enabled.
        return null;
    }
    // Support deprecated function name to preserve compatibility with older versions of VSLS.
    if (!extensionApi.getApi)
        return extensionApi.getApiAsync(liveShareApiVersion);
    return extensionApi.getApi(liveShareApiVersion, callingExtensionId);
}
exports.getApi = getApi;
/** @deprecated */
function getApiAsync() { return getApi(); }
exports.getApiAsync = getApiAsync;
/**
 * A subset of Live Share settings that can be controlled by external policies
 */
var PolicySetting;
(function (PolicySetting) {
    /**
     * Allow guests to start and stop debugging sessions.
     * Mirrors 'Allow Guest Debug Control' setting.
     * Can be either set to true or false.
     */
    PolicySetting["AllowGuestDebugControl"] = "allowGuestDebugControl";
    /**
     * Allow guests to run and terminate workspace tasks
     * Mirrors 'Allow Guest Task Control' setting.
     * Can be either set to true or false.
     */
    PolicySetting["AllowGuestTaskControl"] = "allowGuestTaskControl";
    /**
     * Controls whether web servers are automatically shared.
     * Mirrors 'Auto Share Servers' setting.
     * Can be either set to true or false.
     */
    PolicySetting["AutoShareServers"] = "autoShareServers";
    /**
     * Controls how to handle anonymous join requests.
     * Mirrors 'Anonymous Guest Approval' setting.
     * Can be set to 'prompt'/'accept'/'reject'.
     */
    PolicySetting["AnonymousGuestApproval"] = "anonymousGuestApproval";
    /**
     * Type of connection used for collaboration.
     * Mirrors 'Connection Mode' setting.
     * Can be set to 'auto'/'direct'/'relay'.
     */
    PolicySetting["ConnectionMode"] = "connectionMode";
    /**
     * List of company email-ids allowed to join collaboration sessions.
     */
    PolicySetting["AllowedDomains"] = "allowedDomains";
    /**
     * If set to 'false' enforces read-only terminals in a read/write session.
     * Can be either set to true or false
     */
    PolicySetting["AllowReadWriteTerminals"] = "allowReadWriteTerminals";
})(PolicySetting = exports.PolicySetting || (exports.PolicySetting = {}));
var Role;
(function (Role) {
    Role[Role["None"] = 0] = "None";
    Role[Role["Host"] = 1] = "Host";
    Role[Role["Guest"] = 2] = "Guest";
})(Role = exports.Role || (exports.Role = {}));
/** This is just a placeholder for a richer access control model to be added later. */
var Access;
(function (Access) {
    Access[Access["None"] = 0] = "None";
    Access[Access["ReadOnly"] = 1] = "ReadOnly";
    Access[Access["ReadWrite"] = 3] = "ReadWrite";
    Access[Access["Owner"] = 255] = "Owner";
})(Access = exports.Access || (exports.Access = {}));
/**
 * Identifiers for Live Share tree views. These identifiers may be used by other extensions
 * to extend Live Share tree views with additional nodes via the `registerTreeDataProvider()`
 * API.
 */
var View;
(function (View) {
    View["Session"] = "liveshare.session";
    View["ExplorerSession"] = "liveshare.session.explorer";
    View["PlannedSessions"] = "liveshare.plannedSessions";
    View["Contacts"] = "liveshare.contacts";
    View["Help"] = "liveshare.help";
})(View = exports.View || (exports.View = {}));
/**
 * Identifiers for Live Share tree view items. These identifiers may be used by other
 * extensions to extend Live Share tree items with additional commands using conditional
 * expressions in the `view/item/context` section of their own package.json.
 */
var ViewItem;
(function (ViewItem) {
    // session item groups
    ViewItem["Participants"] = "participants";
    ViewItem["Servers"] = "servers";
    ViewItem["Terminals"] = "terminals";
    ViewItem["Comments"] = "comments";
    ViewItem["Chat"] = "chat";
    // participants
    ViewItem["CurrentUser"] = "participants.currentuser";
    ViewItem["Guest"] = "participants.guest";
    ViewItem["FollowedGuest"] = "participants.guest.followed";
    ViewItem["Participant"] = "participants.participant";
    ViewItem["FollowedParticipant"] = "participants.participant.followed";
    ViewItem["GuestAnonymous"] = "participants.guest.anonymous";
    ViewItem["FollowedGuestAnonymous"] = "participants.guest.followed.anonymous";
    ViewItem["GuestElevated"] = "participants.guest.elevated";
    ViewItem["FollowedGuestElevated"] = "participants.guest.followed.elevated";
    ViewItem["GuestElevatedAnonymous"] = "participants.guest.elevated.anonymous";
    ViewItem["FollowedGuestElevatedAnonymous"] = "participants.guest.followed.elevated.anonymous";
    // servers
    ViewItem["LocalServer"] = "servers.local";
    ViewItem["RemoteServer"] = "servers.remote";
    // terminals
    ViewItem["LocalTerminalReadOnly"] = "terminals.local.readonly";
    ViewItem["LocalTerminalReadWrite"] = "terminals.local.readwrite";
    ViewItem["RemoteTerminal"] = "terminals.remote";
    // contacts
    ViewItem["SuggestedContacts"] = "contacts.suggested";
    ViewItem["AvailableContacts"] = "contacts.available";
    ViewItem["ContactsProvider"] = "contacts.provider";
    ViewItem["SelfContact"] = "contacts.selfContact";
    ViewItem["Contact"] = "contacts.contact";
    ViewItem["ContactInvited"] = "contacts.contact.invited";
    ViewItem["ContactOffline"] = "contacts.contact.offline";
    ViewItem["RecentContact"] = "contacts.recentContact";
    ViewItem["RecentContactOffline"] = "contacts.recentContact.offline";
    ViewItem["RecentContactInvited"] = "contacts.recentContact.invited";
    ViewItem["NoContact"] = "contacts.noContact";
    ViewItem["RecentContacts"] = "contacts.RecentContacts";
    ViewItem["NoSuggestedContacts"] = "contacts.NoSuggestedUsers";
    ViewItem["NoRecentContacts"] = "contacts.NoRecentContacts";
    ViewItem["InvitedContact"] = "contacts.invited";
    // help
    ViewItem["SessionFeedbackQuestion"] = "help.sessionFeedback";
    ViewItem["ReportAProblem"] = "help.reportAProblem";
    ViewItem["TweetUsYourFeedback"] = "help.tweetUsYourFeedback";
    ViewItem["Survey"] = "help.survey";
    ViewItem["GoodFeedback"] = "help.goodFeedback";
    ViewItem["BadFeedback"] = "help.badFeedback";
    ViewItem["DontAskAgain"] = "help.dontAskAgain";
    ViewItem["Thankyou"] = "help.thankyou";
    ViewItem["MoreInfo"] = "help.moreinfo";
    ViewItem["ConfigureSettings"] = "help.configureSettings";
    // Shown while session sharing / joining is in progress
    ViewItem["Loading"] = "loading";
    // Other / unspecified item type
    ViewItem["Other"] = "other";
})(ViewItem = exports.ViewItem || (exports.ViewItem = {}));
class ActivityType {
}
exports.ActivityType = ActivityType;
ActivityType.session = 'session';
ActivityType.workspace = 'workspace';
ActivityType.debug = 'debug';
ActivityType.terminal = 'terminal';
//# sourceMappingURL=liveShare.js.map