"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("stream");
const Deferred_1 = require("./Deferred");
async function writeAsync(stream, chunk) {
    const deferred = new Deferred_1.Deferred();
    stream.write(chunk, (err) => {
        if (err) {
            deferred.reject(err);
        }
        else {
            deferred.resolve();
        }
    });
    return deferred.promise;
}
exports.writeAsync = writeAsync;
function writeSubstream(stream) {
    return new stream_1.Writable({
        async write(chunk, _, callback) {
            try {
                const dv = new DataView(new ArrayBuffer(4));
                dv.setUint32(0, chunk.length, false);
                await writeAsync(stream, Buffer.from(dv.buffer));
                await writeAsync(stream, chunk);
                callback();
            }
            catch (err) {
                callback(err);
            }
        },
        final(callback) {
            // Write the terminating 0 length sequence.
            stream.write(new Uint8Array(4), callback);
        },
    });
}
exports.writeSubstream = writeSubstream;
function readSubstream(stream) {
    return new stream_1.Readable({
        async read(_) {
            const lenBuffer = await getBufferFrom(stream, 4);
            const dv = new DataView(lenBuffer.buffer, lenBuffer.byteOffset, lenBuffer.length);
            const chunkSize = dv.getUint32(0, false);
            if (chunkSize === 0) {
                this.push(null);
                return;
            }
            // TODO: make this *stream* instead of read as an atomic chunk.
            const payload = await getBufferFrom(stream, chunkSize);
            this.push(payload);
        },
    });
}
exports.readSubstream = readSubstream;
async function getBufferFrom(readable, size, allowEndOfStream = false, cancellationToken) {
    const streamEnded = new Deferred_1.Deferred();
    while (size > 0) {
        const readBuffer = readable.read(size);
        if (readBuffer === null) {
            const bytesAvailable = new Deferred_1.Deferred();
            readable.once("readable", bytesAvailable.resolve.bind(bytesAvailable));
            readable.once("end", streamEnded.resolve.bind(streamEnded));
            const endPromise = Promise.race([bytesAvailable.promise, streamEnded.promise]);
            await (cancellationToken ? cancellationToken.racePromise(endPromise) : endPromise);
            if (bytesAvailable.isCompleted) {
                continue;
            }
        }
        if (!allowEndOfStream) {
            if (!readBuffer || readBuffer.length < size) {
                throw new Error("Stream terminated before required bytes were read.");
            }
        }
        return readBuffer;
    }
    return new Buffer([]);
}
exports.getBufferFrom = getBufferFrom;
function throwIfDisposed(value) {
    if (value.isDisposed) {
        throw new Error("disposed");
    }
}
exports.throwIfDisposed = throwIfDisposed;
function requireInteger(parameterName, value, serializedByteLength, signed = "signed") {
    if (!Number.isInteger(value)) {
        throw new Error(`${parameterName} must be an integer.`);
    }
    let bits = serializedByteLength * 8;
    if (signed === "signed") {
        bits--;
    }
    const maxValue = Math.pow(2, bits) - 1;
    const minValue = signed === "signed" ? -Math.pow(2, bits) : 0;
    if (value > maxValue || value < minValue) {
        throw new Error(`${parameterName} must be in the range ${minValue}-${maxValue}.`);
    }
}
exports.requireInteger = requireInteger;
function removeFromQueue(value, queue) {
    if (queue) {
        const idx = queue.indexOf(value);
        if (idx >= 0) {
            queue.splice(idx, 1);
        }
    }
}
exports.removeFromQueue = removeFromQueue;
//# sourceMappingURL=Utilities.js.map