"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoteServiceBroker = void 0;
const assert = require("assert");
const cancellationtoken_1 = require("cancellationtoken");
const events_1 = require("events");
const nerdbank_streams_1 = require("nerdbank-streams");
const AuthorizationServiceClient_1 = require("./AuthorizationServiceClient");
const constants_1 = require("./constants");
const FrameworkServices_1 = require("./FrameworkServices");
const ProxyForIRemoteServiceBroker_1 = require("./ProxyForIRemoteServiceBroker");
const RemoteServiceConnectionInfo_1 = require("./RemoteServiceConnectionInfo");
const utilities_1 = require("./utilities");
/**
 * Facilitates requests to service brokers for services
 */
class RemoteServiceBroker extends events_1.EventEmitter {
    /**
     * Initializes a new instance of the [ServiceBroker](#ServiceBroker) class
     * @param serviceBroker The remote service broker to use for requests
     * @param clientMetadata The client metadata for the remote service broker
     * @param multiplexingStream An optional multiplexing stream to use in making requests
     */
    constructor(serviceBroker, clientMetadata, multiplexingStream) {
        super();
        this.serviceBroker = serviceBroker;
        this.multiplexingStream = multiplexingStream;
        /**
         * Indicates if the service broker has been disposed
         */
        this.isDisposed = false;
        /**
         * Indicates if the service broker owns the multiplexing stream
         */
        this.ownsMxStream = false;
        /**
         * Defines the default client culture used in communication.
         */
        this.defaultClientCulture = 'en-US';
        /**
         * Defines the default client UI culture used in communication.
         */
        this.defaultClientUICulture = 'en-US';
        assert(serviceBroker);
        assert(clientMetadata);
        if ((clientMetadata.supportedConnections & constants_1.RemoteServiceConnections.Multiplexing) === constants_1.RemoteServiceConnections.Multiplexing) {
            assert(multiplexingStream);
        }
        this.serviceBroker = serviceBroker;
        this.serviceBroker.on('availabilityChanged', (args) => {
            this.emit('availabilityChanged', args);
        });
        this.multiplexingStream = multiplexingStream;
    }
    /**
     * Connects to a pipe to a remote service broker that can then answer service requests
     * @param server The pipe to connect to
     * @param cancellationToken A cancellation token
     */
    static async connectToDuplex(server, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(server);
        const rpc = FrameworkServices_1.FrameworkServices.remoteServiceBroker.constructRpcConnection(server);
        const remoteBroker = new ProxyForIRemoteServiceBroker_1.ProxyForIRemoteServiceBroker(rpc.messageConnection);
        return await RemoteServiceBroker.initializeBrokerConnection(remoteBroker, cancellationToken);
    }
    /**
     * Connects to a remote service broker that can answer service requests
     * @param server The remote service broker
     * @param cancellationToken A cancellation token
     */
    static async connectToRemoteServiceBroker(server, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(server);
        return await RemoteServiceBroker.initializeBrokerConnection(server, cancellationToken);
    }
    /**
     * Connects to a multiplexing remote service broker through a pipe
     * @param server The pipe to connect to
     * @param cancellationToken A cancellation token
     */
    static async connectToMultiplexingDuplex(server, options, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(server);
        const multiplexingStream = await nerdbank_streams_1.MultiplexingStream.CreateAsync(server, options, cancellationToken);
        const channel = await multiplexingStream.acceptChannelAsync('', undefined, cancellationToken);
        const connection = FrameworkServices_1.FrameworkServices.remoteServiceBroker.constructRpcConnection(channel);
        const serviceBroker = new ProxyForIRemoteServiceBroker_1.ProxyForIRemoteServiceBroker(connection.messageConnection);
        const result = await this.connectToMultiplexingRemoteServiceBroker(serviceBroker, multiplexingStream, cancellationToken);
        result.ownsMxStream = true;
        return result;
    }
    /**
     * Connects to a multiplexing remote service broker
     * @param server The remote service broker
     * @param stream A multiplexing stream to use in communication
     * @param cancellationToken A cancellation token
     */
    static async connectToMultiplexingRemoteServiceBroker(server, stream, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(server);
        assert(stream);
        const clientMetadata = {
            supportedConnections: constants_1.RemoteServiceConnections.Multiplexing,
        };
        const result = await RemoteServiceBroker.initializeBrokerConnection(server, cancellationToken, clientMetadata, stream);
        result.ownsMxStream = false;
        return result;
    }
    static async initializeBrokerConnection(server, cancellationToken = cancellationtoken_1.default.CONTINUE, clientMetadata, multiplexingStream) {
        clientMetadata = clientMetadata ? clientMetadata : { supportedConnections: constants_1.RemoteServiceConnections.None };
        try {
            await server.handshake(clientMetadata, cancellationToken);
        }
        catch (err) {
            const disposableServer = server;
            disposableServer === null || disposableServer === void 0 ? void 0 : disposableServer.dispose();
            throw err;
        }
        return new RemoteServiceBroker(server, clientMetadata, multiplexingStream);
    }
    /**
     * Sets the authorization service to use when sending requests
     * @param authorizationService The authorization service
     */
    setAuthorizationService(authorizationService) {
        var _a;
        assert(authorizationService);
        (_a = this.authorizationClient) === null || _a === void 0 ? void 0 : _a.dispose();
        this.authorizationClient = new AuthorizationServiceClient_1.AuthorizationServiceClient(authorizationService);
    }
    async getProxy(serviceDescriptor, options, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(serviceDescriptor);
        options = options ? options : { clientCulture: this.defaultClientCulture, clientUICulture: this.defaultClientUICulture };
        options = await this.applyAuthorization(options, cancellationToken);
        let rpc;
        let pipe;
        let remoteConnectionInfo = {};
        try {
            remoteConnectionInfo = await this.serviceBroker.requestServiceChannel(serviceDescriptor.moniker, options, cancellationToken);
            if (!remoteConnectionInfo || RemoteServiceConnectionInfo_1.RemoteServiceConnectionInfo.isEmpty(remoteConnectionInfo)) {
                return null;
            }
            else if (remoteConnectionInfo.multiplexingChannelId && this.multiplexingStream) {
                pipe = await this.multiplexingStream.acceptChannel(remoteConnectionInfo.multiplexingChannelId);
                rpc = serviceDescriptor.constructRpc(options === null || options === void 0 ? void 0 : options.clientRpcTarget, pipe);
            }
            else if (remoteConnectionInfo.pipeName) {
                throw new Error('Cannot connect to named pipe');
            }
            else {
                throw new Error('Unsupported connection type');
            }
        }
        catch (err) {
            rpc === null || rpc === void 0 ? void 0 : rpc.dispose();
            if (utilities_1.IsReadWriteStream(pipe)) {
                pipe.end();
            }
            else if (pipe instanceof nerdbank_streams_1.Channel) {
                pipe.dispose();
            }
            if (!pipe && (remoteConnectionInfo === null || remoteConnectionInfo === void 0 ? void 0 : remoteConnectionInfo.requestId)) {
                await this.serviceBroker.cancelServiceRequest(remoteConnectionInfo.requestId, cancellationToken);
            }
            throw err;
        }
        return rpc;
    }
    async getPipe(serviceMoniker, options, cancellationToken = cancellationtoken_1.default.CONTINUE) {
        assert(serviceMoniker);
        options = options ? options : { clientCulture: 'en-US', clientUICulture: 'en-US' };
        options = await this.applyAuthorization(options, cancellationToken);
        if (options.clientRpcTarget) {
            throw new Error('Cannot connect pipe to service with client RPC target');
        }
        let remoteConnectionInfo = {};
        let pipe = null;
        let channel = null;
        try {
            remoteConnectionInfo = await this.serviceBroker.requestServiceChannel(serviceMoniker, options, cancellationToken);
            if (!remoteConnectionInfo || RemoteServiceConnectionInfo_1.RemoteServiceConnectionInfo.isEmpty(remoteConnectionInfo)) {
                return null;
            }
            if (remoteConnectionInfo.multiplexingChannelId && this.multiplexingStream) {
                channel = await this.multiplexingStream.acceptChannelAsync('', undefined, cancellationToken);
                pipe = channel.stream;
            }
            else if (remoteConnectionInfo.pipeName) {
                throw new Error('Cannot connect to named pipe');
            }
            else {
                throw new Error('Unsupported connection type');
            }
            return pipe;
        }
        catch (err) {
            channel === null || channel === void 0 ? void 0 : channel.dispose();
            pipe === null || pipe === void 0 ? void 0 : pipe.end();
            if (!pipe && (remoteConnectionInfo === null || remoteConnectionInfo === void 0 ? void 0 : remoteConnectionInfo.requestId)) {
                await this.serviceBroker.cancelServiceRequest(remoteConnectionInfo.requestId, cancellationToken);
            }
            throw err;
        }
    }
    dispose() {
        var _a;
        this.isDisposed = true;
        this.serviceBroker.removeAllListeners(constants_1.availabilityChangedEvent);
        const disposableServer = this.serviceBroker;
        disposableServer === null || disposableServer === void 0 ? void 0 : disposableServer.dispose();
        (_a = this.authorizationClient) === null || _a === void 0 ? void 0 : _a.dispose();
        if (this.multiplexingStream && this.ownsMxStream) {
            this.multiplexingStream.dispose();
        }
    }
    async applyAuthorization(options, cancellationToken) {
        if (this.authorizationClient && (!options.clientCredentials)) {
            options.clientCredentials = await this.authorizationClient.getCredentials(cancellationToken);
        }
        return options;
    }
}
exports.RemoteServiceBroker = RemoteServiceBroker;

// SIG // Begin signature block
// SIG // MIIheQYJKoZIhvcNAQcCoIIhajCCIWYCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // B8AYflIIa+x4+stP+bnXywWah02E6yYT0YsDzTkPsiug
// SIG // ggt2MIIE/jCCA+agAwIBAgITMwAAAyauzu35vOR7kgAA
// SIG // AAADJjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTIwMDMwNDE4MjkyOVoX
// SIG // DTIxMDMwMzE4MjkyOVowdDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEeMBwGA1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
// SIG // nZc0xLte0Zp4PVs0bsm9sbjvgdQcrPi/jo5HMBqVz3i/
// SIG // v58maYsTXTnKfo5YVLnOEdTQzdMajP+5bGM0UaIFyUWO
// SIG // t9e0afja/7xLEMQRr/B7aYO5HkhHCTmdhZfGGUI6tk21
// SIG // xW4y/3lnK336zimaOVrPFlXcdUGrqyy8UoBvPbAkOutq
// SIG // HQ17LE7N3G0Xj62tBLgulvRiJGyu3SHrlQ6iUDNcrsi/
// SIG // U2BkvrpZWGHZtEbO0+vXPrryzcJvvu+9InEYRdnLdC3S
// SIG // Fi580IVdmpeUP8RxhDO8jbPKI/B8MZwt6/PlamNT5RDS
// SIG // aQdRbeDBU1p75SOn+/bHlMQpNvl3KxUZzQIDAQABo4IB
// SIG // fTCCAXkwHwYDVR0lBBgwFgYKKwYBBAGCNz0GAQYIKwYB
// SIG // BQUHAwMwHQYDVR0OBBYEFF1zu5HCevZdHtlOyC0v4AOr
// SIG // ptwSMFQGA1UdEQRNMEukSTBHMS0wKwYDVQQLEyRNaWNy
// SIG // b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQx
// SIG // FjAUBgNVBAUTDTIzMDg2NSs0NTg0OTQwHwYDVR0jBBgw
// SIG // FoAU5vxfe7siAFjkck619CF0IzLm76wwVgYDVR0fBE8w
// SIG // TTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29t
// SIG // L3BraS9jcmwvcHJvZHVjdHMvTWljQ29kU2lnUENBXzIw
// SIG // MTAtMDctMDYuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggr
// SIG // BgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29t
// SIG // L3BraS9jZXJ0cy9NaWNDb2RTaWdQQ0FfMjAxMC0wNy0w
// SIG // Ni5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsF
// SIG // AAOCAQEAGebsrQzcccEshd0dcqPjhJA8+0XYpGaevZjO
// SIG // fQpgJWyL4IN6ZVOziQ8iF6TPdfVjYQEVv7dgMnk9/DbP
// SIG // 2nV9kfU0pCSQLpZRAEcGyvi2te6YyrWvET0oEUOHiJkj
// SIG // HbExyhv+sVmWiyP7W2nfM3NQCaXvsqY3Nkl19cr7j9N1
// SIG // 5FMhGpNLgGj2BBbfKO1gkiH6m2/mt9Habf/nO+vKMbyI
// SIG // 1leZaX1g6f2BNoMZaQsTxqVPgZyWcdnkAPZwpDaPDhOu
// SIG // i837rVdRlFtYaqnYmGjJjV9trXpU/GOg9m4zThgOH6AS
// SIG // 65iOjlab05DLZFh7H2dCppHvePf1Lv6y/RUoF12zLTCC
// SIG // BnAwggRYoAMCAQICCmEMUkwAAAAAAAMwDQYJKoZIhvcN
// SIG // AQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNV
// SIG // BAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1
// SIG // dGhvcml0eSAyMDEwMB4XDTEwMDcwNjIwNDAxN1oXDTI1
// SIG // MDcwNjIwNTAxN1owfjELMAkGA1UEBhMCVVMxEzARBgNV
// SIG // BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
// SIG // HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEo
// SIG // MCYGA1UEAxMfTWljcm9zb2Z0IENvZGUgU2lnbmluZyBQ
// SIG // Q0EgMjAxMDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC
// SIG // AQoCggEBAOkOZFB5Z7XE4/0JAEyelKz3VmjqRNjPxVhP
// SIG // qaV2fG1FutM5krSkHvn5ZYLkF9KP/UScCOhlk84sVYS/
// SIG // fQjjLiuoQSsYt6JLbklMaxUH3tHSwokecZTNtX9LtK8I
// SIG // 2MyI1msXlDqTziY/7Ob+NJhX1R1dSfayKi7VhbtZP/iQ
// SIG // tCuDdMorsztG4/BGScEXZlTJHL0dxFViV3L4Z7klIDTe
// SIG // XaallV6rKIDN1bKe5QO1Y9OyFMjByIomCll/B+z/Du2A
// SIG // EjVMEqa+Ulv1ptrgiwtId9aFR9UQucboqu6Lai0FXGDG
// SIG // tCpbnCMcX0XjGhQebzfLGTOAaolNo2pmY3iT1TDPlR8C
// SIG // AwEAAaOCAeMwggHfMBAGCSsGAQQBgjcVAQQDAgEAMB0G
// SIG // A1UdDgQWBBTm/F97uyIAWORyTrX0IXQjMubvrDAZBgkr
// SIG // BgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8EBAMC
// SIG // AYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV
// SIG // 9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEug
// SIG // SaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtp
// SIG // L2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0w
// SIG // Ni0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUF
// SIG // BzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtp
// SIG // L2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNy
// SIG // dDCBnQYDVR0gBIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEw
// SIG // PQYIKwYBBQUHAgEWMWh0dHA6Ly93d3cubWljcm9zb2Z0
// SIG // LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYI
// SIG // KwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwA
// SIG // aQBjAHkAXwBTAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJ
// SIG // KoZIhvcNAQELBQADggIBABp071dPKXvEFoV4uFDTIvwJ
// SIG // nayCl/g0/yosl5US5eS/z7+TyOM0qduBuNweAL7SNW+v
// SIG // 5X95lXflAtTx69jNTh4bYaLCWiMa8IyoYlFFZwjjPzwe
// SIG // k/gwhRfIOUCm1w6zISnlpaFpjCKTzHSY56FHQ/JTrMAP
// SIG // MGl//tIlIG1vYdPfB9XZcgAsaYZ2PVHbpjlIyTdhbQfd
// SIG // UxnLp9Zhwr/ig6sP4GubldZ9KFGwiUpRpJpsyLcfShoO
// SIG // aanX3MF+0Ulwqratu3JHYxf6ptaipobsqBBEm2O2smmJ
// SIG // BsdGhnoYP+jFHSHVe/kCIy3FQcu/HUzIFu+xnH/8IktJ
// SIG // im4V46Z/dlvRU3mRhZ3V0ts9czXzPK5UslJHasCqE5XS
// SIG // jhHamWdeMoz7N4XR3HWFnIfGWleFwr/dDY+Mmy3rtO7P
// SIG // J9O1Xmn6pBYEAackZ3PPTU+23gVWl3r36VJN9HcFT4XG
// SIG // 2Avxju1CCdENduMjVngiJja+yrGMbqod5IXaRzNij6TJ
// SIG // kTNfcR5Ar5hlySLoQiElihwtYNk3iUGJKhYP12E8lGhg
// SIG // Uu/WR5mggEDuFYF3PpzgUxgaUB04lZseZjMTJzkXeIc2
// SIG // zk7DX7L1PUdTtuDl2wthPSrXkizON1o+QEIxpB8QCMJW
// SIG // nL8kXVECnWp50hfT2sGUjgd7JXFEqwZq5tTG3yOalnXF
// SIG // MYIVWzCCFVcCAQEwgZUwfjELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEoMCYGA1UEAxMfTWljcm9zb2Z0IENvZGUgU2lnbmlu
// SIG // ZyBQQ0EgMjAxMAITMwAAAyauzu35vOR7kgAAAAADJjAN
// SIG // BglghkgBZQMEAgEFAKCBrjAZBgkqhkiG9w0BCQMxDAYK
// SIG // KwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYKKwYB
// SIG // BAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgI9N0T3XTj4xm
// SIG // RMTL1Vlt/anes8kW+ka8EKms9xUAQLYwQgYKKwYBBAGC
// SIG // NwIBDDE0MDKgFIASAE0AaQBjAHIAbwBzAG8AZgB0oRqA
// SIG // GGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbTANBgkqhkiG
// SIG // 9w0BAQEFAASCAQA0Djr/0L+waGoF4K7JdcD1jcvdlvS7
// SIG // Gn+zSuRr2T9r2OgM6uqvPq6MWGDBQ6X6kb+V/38WZXFU
// SIG // 015fzQW8164nSu8RWUw+zFvKnly5eLCayuR7BQbtFahr
// SIG // VTiN4Ob4Gd+yzg8Nz7qO5bfyVRaZmfOdHfdWNw5eRK3Q
// SIG // GpV1R+vNwIOFgXkH+CQ+F5hy/WWHYkq8cLvZW2vYCzJB
// SIG // 7VSMhGhXECxNKIKZ53d4xldcB/fdz8o+nyMDJke94Hes
// SIG // p4e/Eik+TqID/v+c4YBziwlCnYSi4NZVdEWJE270vLpi
// SIG // /zZTsm8xCw47swFBP4vqB3cY+S6bL9IAXaEW2zICDosW
// SIG // NEYcoYIS5TCCEuEGCisGAQQBgjcDAwExghLRMIISzQYJ
// SIG // KoZIhvcNAQcCoIISvjCCEroCAQMxDzANBglghkgBZQME
// SIG // AgEFADCCAVEGCyqGSIb3DQEJEAEEoIIBQASCATwwggE4
// SIG // AgEBBgorBgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAE
// SIG // IGOia+w1XjhmdykmW1K714keieto19GioRyDr+4svSIn
// SIG // AgZe1lS3IqQYEzIwMjAwNjExMTg0NTI1LjIyOFowBIAC
// SIG // AfSggdCkgc0wgcoxCzAJBgNVBAYTAlVTMQswCQYDVQQI
// SIG // EwJXQTEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRN
// SIG // aWNyb3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0
// SIG // ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjhENDEt
// SIG // NEJGNy1CM0I3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBTZXJ2aWNloIIOPDCCBPEwggPZoAMCAQIC
// SIG // EzMAAAEKUsg5AVLRcEsAAAAAAQowDQYJKoZIhvcNAQEL
// SIG // BQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
// SIG // bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
// SIG // FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcN
// SIG // MTkxMDIzMjMxOTE1WhcNMjEwMTIxMjMxOTE1WjCByjEL
// SIG // MAkGA1UEBhMCVVMxCzAJBgNVBAgTAldBMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVs
// SIG // YW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQGA1UECxMd
// SIG // VGhhbGVzIFRTUyBFU046OEQ0MS00QkY3LUIzQjcxJTAj
// SIG // BgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZp
// SIG // Y2UwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQC7PhutlKFwxZ+GePU/V0pMke215HSX8PcLX1hjYUby
// SIG // CERBFs7/wEwrbwMIZdOo7NDqcIUhXXt3kxg1OqBJxuoz
// SIG // VcCJ8JwRy/VI79p1ZeLbSv3aMMxouOzoqaNL/Dmb8CT9
// SIG // UEcqq3PF18vMv1cZfk8ZphuVSGPM0eWsJvE1kfPXCJsY
// SIG // zsZturq0jEI6XBh9hpuKQq8KSXvoqCE37EZWrYWy3uhR
// SIG // Jnsrd4Tq2YgYsyWQ/aQF20db73ZWwItXG4TUly4IQ0pc
// SIG // Qi9/UH3fsVu06q8/yNvc7MfIcmnYOUPOyFMBh0EW519K
// SIG // /mg/xYgMhtmZlnzmvHnr5npzJTiwbBuhnwUnAgMBAAGj
// SIG // ggEbMIIBFzAdBgNVHQ4EFgQU+ESUpf06TE1Q3pH4Oq0B
// SIG // opFxhSgwHwYDVR0jBBgwFoAU1WM6XIoxkPNDe3xGG8Uz
// SIG // aFqFbVUwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2Ny
// SIG // bC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMv
// SIG // TWljVGltU3RhUENBXzIwMTAtMDctMDEuY3JsMFoGCCsG
// SIG // AQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNUaW1T
// SIG // dGFQQ0FfMjAxMC0wNy0wMS5jcnQwDAYDVR0TAQH/BAIw
// SIG // ADATBgNVHSUEDDAKBggrBgEFBQcDCDANBgkqhkiG9w0B
// SIG // AQsFAAOCAQEAVJeufNQV8t3TcyWq0Su3nVYZfdRcV6is
// SIG // Tp0Zj5gjBKZ8VEpE3AR7xyYu3QQ7F7PJNXr7991hPKs9
// SIG // w8O+BHeToXmwd4oTGiGOupyPEBrfJVD1IllqRdlUrNod
// SIG // bNu8y4DyRybOPQn9jr+mTntoWyn+Sv6W7lo13DlXdaCK
// SIG // 0linATp+hlCwGtNM81GEhdUwec8STqzb7ucLpPL1ksgm
// SIG // Fh4zKou6K0kYq8SJGEPw9jOQYmcuSOnrUgIOT/TRlVm+
// SIG // +Vcuie2HfZmih5n3/7vrSj2DaVSEXyhoscIHWLzZ1QKF
// SIG // d3Nm6VQTBDkJlaHxYiNBlJS6847W9XQV86p03BwPJe4V
// SIG // 0jCCBnEwggRZoAMCAQICCmEJgSoAAAAAAAIwDQYJKoZI
// SIG // hvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
// SIG // EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
// SIG // HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAw
// SIG // BgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRl
// SIG // IEF1dGhvcml0eSAyMDEwMB4XDTEwMDcwMTIxMzY1NVoX
// SIG // DTI1MDcwMTIxNDY1NVowfDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAg
// SIG // UENBIDIwMTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCpHQ28dxGKOiDs/BOX9fp/aZRrdFQQ1aUK
// SIG // AIKF++18aEssX8XD5WHCdrc+Zitb8BVTJwQxH0EbGpUd
// SIG // zgkTjnxhMFmxMEQP8WCIhFRDDNdNuDgIs0Ldk6zWczBX
// SIG // JoKjRQ3Q6vVHgc2/JGAyWGBG8lhHhjKEHnRhZ5FfgVSx
// SIG // z5NMksHEpl3RYRNuKMYa+YaAu99h/EbBJx0kZxJyGiGK
// SIG // r0tkiVBisV39dx898Fd1rL2KQk1AUdEPnAY+Z3/1ZsAD
// SIG // lkR+79BL/W7lmsqxqPJ6Kgox8NpOBpG2iAg16HgcsOmZ
// SIG // zTznL0S6p/TcZL2kAcEgCZN4zfy8wMlEXV4WnAEFTyJN
// SIG // AgMBAAGjggHmMIIB4jAQBgkrBgEEAYI3FQEEAwIBADAd
// SIG // BgNVHQ4EFgQU1WM6XIoxkPNDe3xGG8UzaFqFbVUwGQYJ
// SIG // KwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQD
// SIG // AgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU
// SIG // 1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBL
// SIG // oEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3Br
// SIG // aS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAt
// SIG // MDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEF
// SIG // BQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3Br
// SIG // aS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5j
// SIG // cnQwgaAGA1UdIAEB/wSBlTCBkjCBjwYJKwYBBAGCNy4D
// SIG // MIGBMD0GCCsGAQUFBwIBFjFodHRwOi8vd3d3Lm1pY3Jv
// SIG // c29mdC5jb20vUEtJL2RvY3MvQ1BTL2RlZmF1bHQuaHRt
// SIG // MEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAFAA
// SIG // bwBsAGkAYwB5AF8AUwB0AGEAdABlAG0AZQBuAHQALiAd
// SIG // MA0GCSqGSIb3DQEBCwUAA4ICAQAH5ohRDeLG4Jg/gXED
// SIG // PZ2joSFvs+umzPUxvs8F4qn++ldtGTCzwsVmyWrf9efw
// SIG // eL3HqJ4l4/m87WtUVwgrUYJEEvu5U4zM9GASinbMQEBB
// SIG // m9xcF/9c+V4XNZgkVkt070IQyK+/f8Z/8jd9Wj8c8pl5
// SIG // SpFSAK84Dxf1L3mBZdmptWvkx872ynoAb0swRCQiPM/t
// SIG // A6WWj1kpvLb9BOFwnzJKJ/1Vry/+tuWOM7tiX5rbV0Dp
// SIG // 8c6ZZpCM/2pif93FSguRJuI57BlKcWOdeyFtw5yjojz6
// SIG // f32WapB4pm3S4Zz5Hfw42JT0xqUKloakvZ4argRCg7i1
// SIG // gJsiOCC1JeVk7Pf0v35jWSUPei45V3aicaoGig+JFrph
// SIG // pxHLmtgOR5qAxdDNp9DvfYPw4TtxCd9ddJgiCGHasFAe
// SIG // b73x4QDf5zEHpJM692VHeOj4qEir995yfmFrb3epgcun
// SIG // Caw5u+zGy9iCtHLNHfS4hQEegPsbiSpUObJb2sgNVZl6
// SIG // h3M7COaYLeqN4DMuEin1wC9UJyH3yKxO2ii4sanblrKn
// SIG // QqLJzxlBTeCG+SqaoxFmMNO7dDJL32N79ZmKLxvHIa9Z
// SIG // ta7cRDyXUHHXodLFVeNp3lfB0d4wwP3M5k37Db9dT+md
// SIG // Hhk4L7zPWAUu7w2gUDXa7wknHNWzfjUeCLraNtvTX4/e
// SIG // dIhJEqGCAs4wggI3AgEBMIH4oYHQpIHNMIHKMQswCQYD
// SIG // VQQGEwJVUzELMAkGA1UECBMCV0ExEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEtMCsGA1UECxMkTWljcm9zb2Z0IElyZWxhbmQg
// SIG // T3BlcmF0aW9ucyBMaW1pdGVkMSYwJAYDVQQLEx1UaGFs
// SIG // ZXMgVFNTIEVTTjo4RDQxLTRCRjctQjNCNzElMCMGA1UE
// SIG // AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIj
// SIG // CgEBMAcGBSsOAwIaAxUAOb12pXHRf+5RrRVyRXbiGmhj
// SIG // 3vmggYMwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYw
// SIG // JAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
// SIG // MjAxMDANBgkqhkiG9w0BAQUFAAIFAOKMsGMwIhgPMjAy
// SIG // MDA2MTEyMTMwMTFaGA8yMDIwMDYxMjIxMzAxMVowdzA9
// SIG // BgorBgEEAYRZCgQBMS8wLTAKAgUA4oywYwIBADAKAgEA
// SIG // AgIZjAIB/zAHAgEAAgIRnzAKAgUA4o4B4wIBADA2Bgor
// SIG // BgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMCoAowCAIB
// SIG // AAIDB6EgoQowCAIBAAIDAYagMA0GCSqGSIb3DQEBBQUA
// SIG // A4GBADosTpGkj4o8xWNjpc2mqXZgahm+z4a3FynCQA0w
// SIG // MewdLi9Fvhzi5c6SCUNx6r+69c4sfFJ454zv09R1T2vw
// SIG // 0vC5RghyBixuPaG1hICTpCq1FHfDESlGNwNFhoFg2/BU
// SIG // bTa9C3rX0eBPBBMY1a9kwXpHZTrmmMfmA6A7mg74JLNr
// SIG // MYIDDTCCAwkCAQEwgZMwfDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAg
// SIG // UENBIDIwMTACEzMAAAEKUsg5AVLRcEsAAAAAAQowDQYJ
// SIG // YIZIAWUDBAIBBQCgggFKMBoGCSqGSIb3DQEJAzENBgsq
// SIG // hkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQgkp/G/ZM8
// SIG // dc3ZnXeHs+fbp2MRz8qHmyZK6XvsQlYFCQ8wgfoGCyqG
// SIG // SIb3DQEJEAIvMYHqMIHnMIHkMIG9BCBXAzYkM7qhDCgN
// SIG // 6EbxXbZtR3HNkNZaGSMYHzfL5NKsqjCBmDCBgKR+MHwx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
// SIG // c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAABClLI
// SIG // OQFS0XBLAAAAAAEKMCIEIKAPczM7jEhdxE09kU3ATxse
// SIG // 1c3FyWwGVDzQbpLjdWqwMA0GCSqGSIb3DQEBCwUABIIB
// SIG // AIHiq2xfJ5Kmjph4plWWI2IzUnxlJoGP2XgY/s8EC7sm
// SIG // N1Szeol3sU5AyW8MvAfv109pkApEQ4bbIxK5yaEqFf9T
// SIG // GKCq06xPo3zJoJCglm592UoUffZyFfsH4RIMh5vGPOfC
// SIG // vwDGXVfqjf+kcN6r0ex4x/IzGG9uNOP12bkP0Li5N+Pq
// SIG // p9Q2hYw3JsR0uLMyNIjct0Vrf1OWOu/HMWlU/kEJQr+t
// SIG // Z7UFQF3MbDnsPvhw3U3IGSUPcc5ETVWjTsA6XDk0MeIo
// SIG // AG0DN8D2Z4DakZNaD/f5p985wUd6Lm3QtGcK3J4AVftT
// SIG // OsE9EPHlQwiSOEo6If0H24lmLufbCkxkhAg=
// SIG // End signature block
