import torch
import torch.nn.functional as nnf
import cv2
import numpy as np
import os
os.environ["OPENCV_IO_ENABLE_OPENEXR"]="1"
# -----------------------
# Helpers
# -----------------------
def load_exr(path: str, device='cuda') -> torch.Tensor:
    """
    Load an HDR environment map in .exr format as a PyTorch tensor [1,3,H,W], RGB, float32.
    """
    img = cv2.imread(path, cv2.IMREAD_UNCHANGED)
    img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
    tensor = torch.from_numpy(img).permute(2,0,1).unsqueeze(0).float().to(device)
    return tensor  # [1,3,H,W]


def load_texture(path: str, device='cuda', gray=False) -> torch.Tensor:
    """
    Load a texture into [1,C,H,W], normalized [0,1].
    If gray=True, load as single-channel grayscale.
    Otherwise, load as RGB, dropping alpha if present.
    """
    img = cv2.imread(path, cv2.IMREAD_UNCHANGED)
    if gray:
        if img.ndim == 3:
            img = img[..., 0]
        if img.dtype == np.uint8:
            img = img.astype(np.float32) / 255.0
        tensor = torch.from_numpy(img).unsqueeze(0).unsqueeze(0).to(device)
    else:
        if img.ndim == 3 and img.shape[2] == 4:
            img = img[..., :3]
        if img.dtype == np.uint8:
            img = img.astype(np.float32) / 255.0
        if img.ndim == 3 and img.shape[2] == 3:
            img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
        tensor = torch.from_numpy(img).permute(2,0,1).unsqueeze(0).to(device)
    return tensor


def sample_env_map(env: torch.Tensor, dirs: torch.Tensor) -> torch.Tensor:
    """
    Manual bilinear sampling of an equirectangular environment map.
    env: [1,3,H,W]
    dirs: [N,3] normalized direction vectors
    Returns: [N,3]
    """
    # Unpack
    N = dirs.shape[0]
    device = dirs.device
    C, H, W = env.shape[1:]

    # Spherical coords
    x, y, z = dirs[:,0], dirs[:,1], dirs[:,2]
    theta = torch.acos(y.clamp(-1,1))            # [0, pi]
    phi = torch.atan2(z, x)                      # [-pi, pi]
    u = (phi + np.pi) / (2 * np.pi)              # [0,1]
    v = theta / np.pi                            # [0,1]

    # Pixel coordinates
    px = u * (W - 1)
    py = v * (H - 1)

    # Integer coords
    ix0 = torch.floor(px).long().clamp(0, W-1)
    iy0 = torch.floor(py).long().clamp(0, H-1)
    ix1 = (ix0 + 1) % W
    iy1 = torch.clamp(iy0 + 1, 0, H-1)

    # Fractions
    dx = (px - ix0.float()).unsqueeze(1)  # [N,1]
    dy = (py - iy0.float()).unsqueeze(1)

    # Fetch texels
    env_flat = env[0]  # [3,H,W]
    c00 = env_flat[:, iy0, ix0]  # [3,N]
    c10 = env_flat[:, iy0, ix1]
    c01 = env_flat[:, iy1, ix0]
    c11 = env_flat[:, iy1, ix1]

    # Weights
    w00 = (1 - dx) * (1 - dy)
    w10 = dx * (1 - dy)
    w01 = (1 - dx) * dy
    w11 = dx * dy

    # Combine
    result = (c00 * w00.t() + c10 * w10.t() + c01 * w01.t() + c11 * w11.t()).permute(1,0)
    return result  # [N,3]

def save_tensor_img(tensor, filename, is_color=True, H=None, W=None):
    arr = tensor.detach().cpu()
    if arr.ndim == 2:
        arr = arr.unsqueeze(0)
    if arr.ndim == 1:
        arr = arr.unsqueeze(0).unsqueeze(0)
    if H is not None and W is not None:
        arr = arr.view(-1, H, W)
    if is_color:
        if arr.shape[0] == 1:
            arr = arr.repeat(3, 1, 1)
        arr = arr.permute(1, 2, 0).numpy()  # HWC
        arr = np.clip(arr, 0, 1)
        arr = (arr * 255).astype(np.uint8)
        arr = cv2.cvtColor(arr, cv2.COLOR_RGB2BGR)
    else:
        arr = arr.squeeze(0).numpy()  # HW
        arr = np.clip(arr, 0, 1)
        arr = (arr * 255).astype(np.uint8)
    cv2.imwrite(filename, arr)

# -----------------------
# Disney BRDF components
# -----------------------
def fresnel_schlick(cos_theta: torch.Tensor, F0: torch.Tensor) -> torch.Tensor:
    return F0 + (1 - F0) * (1 - cos_theta).clamp(0,1).pow(5)


def distribution_ggx(n_dot_h: torch.Tensor, roughness: torch.Tensor) -> torch.Tensor:
    a = roughness ** 2
    a2 = a * a
    denom = (n_dot_h**2) * (a2 - 1) + 1
    return a2 / (np.pi * denom**2 + 1e-7)


def geometry_schlick_ggx(n_dot_v: torch.Tensor, roughness: torch.Tensor) -> torch.Tensor:
    r = (roughness + 1)**2 / 8
    return n_dot_v / (n_dot_v * (1 - r) + r + 1e-7)


def geometry_smith(n: torch.Tensor, v: torch.Tensor, l: torch.Tensor, roughness: torch.Tensor) -> torch.Tensor:
    n_dot_v = (n * v).sum(-1, keepdim=True).clamp(0,1)
    n_dot_l = (n * l).sum(-1, keepdim=True).clamp(0,1)
    ggx_v = geometry_schlick_ggx(n_dot_v, roughness)
    ggx_l = geometry_schlick_ggx(n_dot_l, roughness)
    return ggx_v * ggx_l

def linear2srgb(x):
    # Convert from linear color space to sRGB
    # x should be in range [0,1]
    return torch.where(x <= 0.0031308,
                     12.92 * x,
                     1.055 * torch.pow(x, 1/2.4) - 0.055)
# -----------------------
# IBL Pipeline
# -----------------------
def render_pbr(
    albedo: torch.Tensor,
    normal_map: torch.Tensor,
    roughness_map: torch.Tensor,
    metallic_map: torch.Tensor,
    env_map: torch.Tensor
) -> torch.Tensor:
    device = albedo.device
    _, _, H, W = albedo.shape

    # Flatten
    n = normal_map.permute(0,2,3,1).reshape(-1,3) * 2 - 1
    n = nnf.normalize(n, dim=-1)
    # n[..., 2] = -n[..., 2]  # Flip z for left-handed to right-handed
    N = n.shape[0]
    v = torch.tensor([0,0,1], device=device).expand(N,3)
    rough = roughness_map.view(1,1,H*W).permute(0,2,1).reshape(N,1)
    metall = metallic_map.view(1,1,H*W).permute(0,2,1).reshape(N,1)
    albedo_flat = albedo.reshape(1,3,H*W).permute(0,2,1).reshape(N,3)

    # F0
    F0 = 0.04 * (1 - metall) + albedo_flat * metall

    # Environment sampling
    # Downsample and blur environment map
    env_map_small = nnf.interpolate(env_map, size=(16, 32), mode='bilinear')
    kernel_size = 3
    sigma = 1.0
    kernel = torch.tensor([
        [0.0625, 0.125, 0.0625],
        [0.125, 0.25, 0.125],
        [0.0625, 0.125, 0.0625]
    ], device=env_map.device).view(1, 1, kernel_size, kernel_size)
    kernel = kernel.expand(3, 1, kernel_size, kernel_size)
    padding = kernel_size // 2
    env_map_blurred = nnf.conv2d(
        env_map_small,
        kernel.to(env_map.dtype),
        padding=padding,
        groups=3
    )
    env_map = torch.clamp(env_map_blurred, max=10.0) / 10
    diff_env = sample_env_map(env_map, n)
    r = nnf.normalize(-v + 2 * (n * (n * v).sum(-1, keepdim=True)), dim=-1)
    spec_env = sample_env_map(env_map, r)

    # BRDF
    h = nnf.normalize(v + r, dim=-1)
    n_dot_h = (n * h).sum(-1, keepdim=True).clamp(0,1)
    n_dot_v = (n * v).sum(-1, keepdim=True).clamp(0,1)
    n_dot_l = (n * r).sum(-1, keepdim=True).clamp(0,1)
    D = distribution_ggx(n_dot_h, rough)
    G = geometry_smith(n, v, r, rough)
    F = fresnel_schlick((h * v).sum(-1, keepdim=True).clamp(0,1), F0)
    spec = (D * G * F) / (4 * n_dot_v * n_dot_l + 1e-7)
    spec = spec * spec_env
    kD = (1 - F) * (1 - metall)
    diff = kD * albedo_flat / np.pi * diff_env
    # Visualization of intermediate results
    save_dir = "intermediate"
    os.makedirs(save_dir, exist_ok=True)
    # Save input maps
    save_tensor_img(albedo[0], f"{save_dir}/albedo.png", is_color=True)
    save_tensor_img(normal_map[0], f"{save_dir}/normal.png", is_color=True)
    save_tensor_img(roughness_map[0], f"{save_dir}/roughness.png", is_color=False)
    save_tensor_img(metallic_map[0], f"{save_dir}/metallic.png", is_color=False)
    # Save intermediate results (reshape to [3,H,W] or [1,H,W] as needed)
    save_tensor_img(F0.t().reshape(3,H,W), f"{save_dir}/F0.png", is_color=True)
    save_tensor_img(torch.clamp(diff_env.t().reshape(3,H,W),0,1), f"{save_dir}/diff_env.png", is_color=True)
    save_tensor_img(torch.clamp(spec_env.t().reshape(3,H,W),0,1), f"{save_dir}/spec_env.png", is_color=True)
    save_tensor_img(torch.clamp(diff.t().reshape(3,H,W),0,1), f"{save_dir}/diff.png", is_color=True)
    save_tensor_img(torch.clamp(spec.t().reshape(3,H,W),0,1), f"{save_dir}/spec.png", is_color=True)
    save_tensor_img(torch.clamp(kD.t().reshape(3,H,W),0,1), f"{save_dir}/kD.png", is_color=True)
    save_tensor_img(torch.clamp(F.t().reshape(3,H,W),0,1), f"{save_dir}/F.png", is_color=True)

    # Combine
    # We need to apply linear2srgb to the final output

    out = (diff + spec).clamp(0,1)
    return out.reshape(H, W, 3).permute(2, 0, 1)

# -----------------------
# Example Usage
# -----------------------
if __name__ == "__main__":
    device = 'cuda' if torch.cuda.is_available() else 'cpu'
    albedo = load_texture('albedo_000_000.png', device=device, gray=False)
    normal = load_texture('normal_000_000.png', device=device, gray=False)
    roughness = load_texture('roughness.png', device=device, gray=True)
    metallic = load_texture('metallic.png', device=device, gray=True)
    env = load_exr('environment.exr', device=device)
    out = render_pbr(albedo, normal, roughness, metallic, env)
    img = out.squeeze(0).permute(1,2,0).cpu().numpy()
    cv2.imwrite('render.png', cv2.cvtColor((img * 255).astype(np.uint8), cv2.COLOR_RGB2BGR))
