from diffusers.models.modeling_utils import ModelMixin
from diffusers import (
    CogVideoXTransformer3DModel,
)

import os
import torch
import torch.nn as nn
import torch.nn.functional as F
from typing import Any, Dict, Optional, Tuple, Union, List, Callable


from diffusers.models.modeling_utils import ModelMixin
from diffusers import CogVideoXTransformer3DModel
from diffusers.utils import USE_PEFT_BACKEND, is_torch_version, logging, scale_lora_layers, unscale_lora_layers
from diffusers.utils.torch_utils import maybe_allow_in_graph
from diffusers.models.modeling_outputs import Transformer2DModelOutput
from diffusers.models.embeddings import TimestepEmbedding, apply_rotary_emb
from diffusers.models.attention import FeedForward, Attention


import os
import torch
import torch.nn as nn
from typing import Any, Dict, Optional, Tuple, Union, List, Callable


class CogVideoXTransformer3DModelSBSFullSeqClean(CogVideoXTransformer3DModel, ModelMixin):
    """
    Expand the input dimension of the vanilla CogVideoX, control the output by domain embeddings

    Parameters:
        num_domain (`int`, default to `2`):
            the number of different domains for the input
        in_channels (`int`, default to `48`):
            The input channels, typically equals to num_domain x dim_each_domain
    """
    def __init__(
        self,
        num_domain: int = 2,
        num_attention_heads: int = 30,
        attention_head_dim: int = 64,
        in_channels: int = 32,
        out_channels: Optional[int] = 16,
        flip_sin_to_cos: bool = True,
        freq_shift: int = 0,
        time_embed_dim: int = 512,
        text_embed_dim: int = 4096,
        num_layers: int = 30,
        dropout: float = 0.0,
        attention_bias: bool = True,
        sample_width: int = 90,
        sample_height: int = 60,
        sample_frames: int = 49,
        patch_size: int = 2,
        temporal_compression_ratio: int = 4,
        max_text_seq_length: int = 226,
        activation_fn: str = "gelu-approximate",
        timestep_activation_fn: str = "silu",
        norm_elementwise_affine: bool = True,
        norm_eps: float = 1e-5,
        spatial_interpolation_scale: float = 1.875,
        temporal_interpolation_scale: float = 1.0,
        use_rotary_positional_embeddings: bool = False,
        use_learned_positional_embeddings: bool = False,
        **kwargs
    ):
        super().__init__(
            num_attention_heads=num_attention_heads,
            attention_head_dim=attention_head_dim,
            in_channels=in_channels,
            out_channels=out_channels,
            flip_sin_to_cos=flip_sin_to_cos,
            freq_shift=freq_shift,
            time_embed_dim=time_embed_dim,
            text_embed_dim=text_embed_dim,
            num_layers=num_layers,
            dropout=dropout,
            attention_bias=attention_bias,
            sample_width=sample_width,
            sample_height=sample_height,
            sample_frames=sample_frames,
            patch_size=patch_size,
            temporal_compression_ratio=temporal_compression_ratio,
            max_text_seq_length=max_text_seq_length,
            activation_fn=activation_fn,
            timestep_activation_fn=timestep_activation_fn,
            norm_elementwise_affine=norm_elementwise_affine,
            norm_eps=norm_eps,
            spatial_interpolation_scale=spatial_interpolation_scale,
            temporal_interpolation_scale=temporal_interpolation_scale,
            use_rotary_positional_embeddings=use_rotary_positional_embeddings,
            use_learned_positional_embeddings=use_learned_positional_embeddings,
            **kwargs
        )
        
        assert num_domain == 2
        self.num_domain = num_domain
        self.domain_embedding = TimestepEmbedding(self.num_domain * 2, time_embed_dim, timestep_activation_fn)
    
    def forward(
        self,
        hidden_states: torch.Tensor, # latent_model_input
        encoder_hidden_states: torch.Tensor, # prompt_embeds
        timestep: Union[int, float, torch.LongTensor],
        timestep_cond: Optional[torch.Tensor] = None,
        ofs: Optional[Union[int, float, torch.LongTensor]] = None,
        image_rotary_emb: Optional[Tuple[torch.Tensor, torch.Tensor]] = None,
        attention_kwargs: Optional[Dict[str, Any]] = None,
        return_dict: bool = True,
        domain_idx: int = None,
        latent_pm: Optional[torch.Tensor] = None,  # Added parameter for latent_pm
    ):
        if attention_kwargs is not None:
            attention_kwargs = attention_kwargs.copy()
            lora_scale = attention_kwargs.pop("scale", 1.0)
        else:
            lora_scale = 1.0

        if USE_PEFT_BACKEND:
            # weight the lora layers by setting `lora_scale` for each PEFT layer
            scale_lora_layers(self, lora_scale)
        else:
            if attention_kwargs is not None and attention_kwargs.get("scale", None) is not None:
                print(
                    "Passing `scale` via `attention_kwargs` when not using the PEFT backend is ineffective."
                )

        batch_size, num_frames, channels, height, width = hidden_states.shape

        # 1. Time embedding
        timesteps = timestep
        t_emb = self.time_proj(timesteps)

        # timesteps does not contain any weights and will always return f32 tensors
        # but time_embedding might actually be running in fp16. so we need to cast here.
        # there might be better ways to encapsulate this.
        t_emb = t_emb.to(dtype=hidden_states.dtype)
        emb = self.time_embedding(t_emb, timestep_cond)

        # 1. Domain embedding
        # domain_emb [bs, 512]
        domain_emb = self.domain_embedding(domain_idx.to(dtype=hidden_states.dtype))

        if self.ofs_embedding is not None:
            ofs_emb = self.ofs_proj(ofs)
            ofs_emb = ofs_emb.to(dtype=hidden_states.dtype)
            ofs_emb = self.ofs_embedding(ofs_emb)
            emb = emb + ofs_emb

        # 2. Patch embedding
        hidden_states = self.patch_embed(encoder_hidden_states, hidden_states)
        hidden_states = self.embedding_dropout(hidden_states)
        text_seq_length = encoder_hidden_states.shape[1]

        # Apply patch embedding to latent_pm
        latent_pm_embedded = self.patch_embed(encoder_hidden_states, latent_pm)
        latent_pm_embedded = self.embedding_dropout(latent_pm_embedded)
        # Extract the part after text_seq_length
        latent_pm_embedded = latent_pm_embedded[:, text_seq_length:]

        encoder_hidden_states = hidden_states[:, :text_seq_length] # [BS, 226, 3072]
        hidden_states = hidden_states[:, text_seq_length:] #[BS, 17550, 3072]
        
        # Concatenate along sequence dimension (dim=1)
        hidden_states = torch.cat([hidden_states, latent_pm_embedded], dim=1)

        # 3. Transformer blocks
        for i, block in enumerate(self.transformer_blocks):
            if torch.is_grad_enabled() and self.gradient_checkpointing:

                def create_custom_forward(module):
                    def custom_forward(*inputs):
                        return module(*inputs)

                    return custom_forward

                ckpt_kwargs: Dict[str, Any] = {"use_reentrant": False} if is_torch_version(">=", "1.11.0") else {}
                hidden_states, encoder_hidden_states = torch.utils.checkpoint.checkpoint(
                    create_custom_forward(block),
                    hidden_states,
                    encoder_hidden_states,
                    emb,
                    domain_emb,
                    image_rotary_emb,
                    **ckpt_kwargs,
                )
            else:
                hidden_states, encoder_hidden_states = block(
                    hidden_states=hidden_states,
                    encoder_hidden_states=encoder_hidden_states,
                    temb=emb,
                    domain_emb=domain_emb,
                    image_rotary_emb=image_rotary_emb,
                )

        if not self.config.use_rotary_positional_embeddings:
            # CogVideoX-2B
            hidden_states = self.norm_final(hidden_states)
        else:
            # CogVideoX-5B
            hidden_states = torch.cat([encoder_hidden_states, hidden_states], dim=1)
            hidden_states = self.norm_final(hidden_states)
            hidden_states = hidden_states[:, text_seq_length:]

        # 4. Final block
        hidden_states = self.norm_out(hidden_states, temb=emb)
        hidden_states = self.proj_out(hidden_states)

        # 5. Unpatchify
        p = self.config.patch_size
        p_t = self.config.patch_size_t

        if p_t is None:
            output_domain_0 = hidden_states[:, :hidden_states.shape[1] // 2, :].reshape(batch_size, num_frames, height // p, width // p, -1, p, p)
            output_domain_1 = hidden_states[:, hidden_states.shape[1] // 2:, :].reshape(batch_size, num_frames, height // p, width // p, -1, p, p)
            output_domain_0 = output_domain_0.permute(0, 1, 4, 2, 5, 3, 6).flatten(5, 6).flatten(3, 4)
            output_domain_1 = output_domain_1.permute(0, 1, 4, 2, 5, 3, 6).flatten(5, 6).flatten(3, 4)
            output = torch.cat([output_domain_0, output_domain_1], dim=1)
        else:
            output_domain_0 = hidden_states[:, :hidden_states.shape[1] // 2, :].reshape(
                batch_size, (num_frames + p_t - 1) // p_t, height // p, width // p, -1, p_t, p, p
            )
            output_domain_1 = hidden_states[:, hidden_states.shape[1] // 2:, :].reshape(
                batch_size, (num_frames + p_t - 1) // p_t, height // p, width // p, -1, p_t, p, p
            )
            output_domain_0 = output_domain_0.permute(0, 1, 5, 4, 2, 6, 3, 7).flatten(6, 7).flatten(4, 5).flatten(1, 2)
            output_domain_1 = output_domain_1.permute(0, 1, 5, 4, 2, 6, 3, 7).flatten(6, 7).flatten(4, 5).flatten(1, 2)
            output = torch.cat([output_domain_0, output_domain_1], dim=1)

        if USE_PEFT_BACKEND:
            # remove `lora_scale` from each PEFT layer
            unscale_lora_layers(self, lora_scale)

        if not return_dict:
            return (output,)
        return Transformer2DModelOutput(sample=output)

    @classmethod
    def from_pretrained(cls, pretrained_model_name_or_path: Optional[Union[str, os.PathLike]], **kwargs):
        try:
            model = super().from_pretrained(pretrained_model_name_or_path, **kwargs)
            print("Loaded Custom Model checkpoint directly.")
            
            return model
        
        except Exception as e:
            print(f"Failed to load as Custom Model: {e}")
            print("Attempting to load as CogVideoXTransformer3DModel and convert...")
            base_model = CogVideoXTransformer3DModel.from_pretrained(pretrained_model_name_or_path, **kwargs)
            
            config = dict(base_model.config)
            config["num_domain"] = kwargs.pop("num_domain", 2)
            
            model = cls(**config)
            model.load_state_dict(base_model.state_dict(), strict=False)            
            return model

    def save_pretrained(
        self,
        save_directory: Union[str, os.PathLike],
        is_main_process: bool = True,
        save_function: Optional[Callable] = None,
        safe_serialization: bool = True,
        variant: Optional[str] = None,
        max_shard_size: Union[int, str] = "5GB",
        push_to_hub: bool = False,
        **kwargs,
    ):
        super().save_pretrained(
            save_directory,
            is_main_process=is_main_process,
            save_function=save_function,
            safe_serialization=safe_serialization,
            variant=variant,
            max_shard_size=max_shard_size,
            push_to_hub=push_to_hub,
            **kwargs,
        )
        
        if is_main_process:
            config_dict = dict(self.config)
            config_dict.pop("_name_or_path", None)
            config_dict.pop("_use_default_values", None)
            config_dict["_class_name"] = "CogVideoXTransformer3DModelSBSFullSeqClean"
            config_dict["num_domain"] = self.num_domain

            
            os.makedirs(save_directory, exist_ok=True)
            with open(os.path.join(save_directory, "config.json"), "w", encoding="utf-8") as f:
                import json
                json.dump(config_dict, f, indent=2)
