import numpy as np
import imageio

def generate_uniform_pointmap(width, height, depth_value=1.0, output_path='pointmap_xyz.png'):
    # Generate a grid of X and Y coordinates
    x = np.linspace(-1, 1, width)
    y = np.linspace(-1, 1, height)
    xv, yv = np.meshgrid(x, y)
    
    # Create a spatially varying Z value: increases from bottom (row 0) to top (row height-1)
    zv = 1 - np.linspace(0, 1, height)[:, None]  # shape (H, 1)
    zv = np.repeat(zv, width, axis=1)        # shape (H, W)
    
    # Stack to get (H, W, 3) array
    pointmap = np.stack([xv, yv, zv], axis=-1)
    
    # Normalize XYZ to [0, 1] for image saving
    # X and Y are already in [-1, 1], so map to [0, 1]
    pointmap[..., 0] = (pointmap[..., 0] + 1)
    pointmap[..., 1] = (pointmap[..., 1] + 1)
    # Z is constant, but ensure it's in [0, 1]
    pointmap[..., 2] = (pointmap[..., 2] - np.min(pointmap[..., 2])) / (np.ptp(pointmap[..., 2]) + 1e-8)
    
    # Convert to 8-bit RGB
    img = (pointmap * 255).astype(np.uint8)
    
    # Save as PNG
    imageio.imwrite(output_path, img)
    print(f"Saved pointmap as {output_path}")

if __name__ == "__main__":
    generate_uniform_pointmap(width=512, height=288, depth_value=2, output_path='pointmap_xyz.png')
