import os
import glob
import torch
from safetensors.torch import load_file
from tqdm import tqdm

def get_tensors_from_data(data):
    """
    Extracts and flattens tensors from the loaded data.
    If the data is a tensor, it is flattened and returned as a list.
    If it is a dict, any tensor values are flattened and returned.
    """
    tensors = []
    if isinstance(data, torch.Tensor):
        tensors.append(data.view(-1))
    elif isinstance(data, dict):
        for key, value in data.items():
            if isinstance(value, torch.Tensor):
                tensors.append(value.view(-1))
    return tensors

def process_folder(folder_path, extension, loader_func):
    """
    Processes all files with a given extension in folder_path using the provided loader function.
    Aggregates all tensor values and computes statistics over them.
    """
    file_pattern = os.path.join(folder_path, f"*{extension}")
    file_list = glob.glob(file_pattern)
    if not file_list:
        print(f"No files with extension '{extension}' found in {folder_path}")
        return

    all_tensors = []
    for file in tqdm(file_list[:5000]):
        try:
            data = loader_func(file)
            tensors = get_tensors_from_data(data)
            if not tensors:
                print(f"No tensors found in file: {file}")
            else:
                all_tensors.extend(tensors)
                import pdb; pdb.set_trace()
        except Exception as e:
            print(f"Error processing file {file}: {e}")

    if not all_tensors:
        print("No tensors collected from the entire folder.")
        return

    # Concatenate all tensors into one long 1D tensor
    concatenated = torch.cat(all_tensors)
    count = concatenated.numel()
    mean = concatenated.mean().item()
    std = concatenated.std().item()
    min_val = concatenated.min().item()
    max_val = concatenated.max().item()

    print(f"\nStatistics for folder '{folder_path}' ({len(file_list)} files):")
    print(f"  Total elements: {count}")
    print(f"  Mean: {mean:.4f}")
    print(f"  Std: {std:.4f}")
    print(f"  Min: {min_val}")
    print(f"  Max: {max_val}")

def load_pt(file_path):
    """Loads a .pt file using torch.load."""
    return torch.load(file_path, map_location="cpu")

def load_safetensors(file_path):
    """Loads a .safetensors file using safetensors.torch.load_file."""
    return load_file(file_path)

def main():
    import argparse
    parser = argparse.ArgumentParser(description="Load tensor files and compute statistics for each folder.")
    parser.add_argument("--pt_folder", type=str, required=True,
                        help="Path to folder containing .pt files")
    parser.add_argument("--safe_folder", type=str, required=True,
                        help="Path to folder containing .safetensors files")
    args = parser.parse_args()

    print("Processing .pt files...")
    process_folder(args.pt_folder, ".pt", load_pt)

    print("\nProcessing .safetensors files...")
    process_folder(args.safe_folder, ".safetensors", load_safetensors)

if __name__ == "__main__":
    main()
