def merge_datalists(datalist1_path, datalist2_path, output_path, separator='|'):
    """
    Merge two datalist files where each line in the output file contains
    one entry from datalist1 and one entry from datalist2, separated by a special character.
    
    Args:
        datalist1_path (str): Path to the first datalist file
        datalist2_path (str): Path to the second datalist file
        output_path (str): Path to save the merged datalist
        separator (str): Special character to separate entries
    
    Returns:
        int: Number of merged entries written to the output file
    """
    # Read the first datalist
    with open(datalist1_path, 'r') as f:
        datalist1 = [line.strip() for line in f.readlines()]
    
    # Read the second datalist
    with open(datalist2_path, 'r') as f:
        datalist2 = [line.strip() for line in f.readlines()]
    
    # Calculate how many entries to process
    min_length = min(len(datalist1), len(datalist2))
    
    # Create merged datalist
    merged_entries = []
    for i in range(min_length):
        merged_entry = f"{datalist1[i]}{separator}{datalist2[i]}"
        merged_entries.append(merged_entry)
    
    # Write the merged datalist to output file
    with open(output_path, 'w') as f:
        for entry in merged_entries:
            f.write(entry + '\n')
    
    return len(merged_entries)


if __name__ == "__main__":
    import argparse
    
    parser = argparse.ArgumentParser(description="Merge two datalists into one")
    parser.add_argument("datalist1", type=str, help="Path to the first datalist file")
    parser.add_argument("datalist2", type=str, help="Path to the second datalist file")
    parser.add_argument("output", type=str, help="Path to save the merged datalist")
    parser.add_argument("--separator", type=str, default="|", help="Separator character between entries (default: |)")
    
    args = parser.parse_args()
    
    count = merge_datalists(args.datalist1, args.datalist2, args.output, args.separator)
    print(f"Successfully merged {count} entries into {args.output}") 