import os
import hashlib
import numpy as np
import imageio
import torch
from pathlib import Path
import torchvision.transforms.functional as F
import torch.nn.functional as nnf
import argparse
from core.dataset import PointmapDataset
from core.tokenizer.cogvideo import CogvideoTokenizer
from transformers import AutoTokenizer, T5EncoderModel
from safetensors.torch import save_file


def split_list(datalist, num_segments):
    """Splits list datalist into num_segments continuous segments with balanced load."""
    n = len(datalist)
    segment_size = n // num_segments
    remainder = n % num_segments  # Extra elements to distribute

    segments = []
    start = 0
    for i in range(num_segments):
        extra = 1 if i < remainder else 0  # Distribute remainder among the first few segments
        end = start + segment_size + extra
        segments.append(datalist[start:end])
        start = end

    return segments


def encode_video(video, vae):
    """Encode video using VAE, following lora_trainer encode_video method."""
    # # Convert video to torch tensor if it's numpy array
    # if isinstance(video, np.ndarray):
    #     video = torch.from_numpy(video).float()
    #     # Normalize to [-1, 1] if in [0, 255] range
    #     if video.max() > 1.0:
    #         video = video / 255.0 * 2.0 - 1.0
        
    #     # Ensure video is in [B, C, F, H, W] format
    #     if video.ndim == 4:  # [F, C, H, W]
    #         video = video.unsqueeze(0)
    #     elif video.ndim == 5 and video.shape[0] != 1:  # [F, B, C, H, W]
    #         video = video.permute(1, 2, 0, 3, 4)
    
    # Move to VAE's device and dtype
    # video = video.to(vae.device, dtype=vae.dtype)
    # video = video.permute(0, 2, 1, 3, 4).contiguous()
    
    # Encode using VAE
    with torch.no_grad():
        import pdb; pdb.set_trace()
        latent = vae.encode(video.numpy())[0]
        latent = latent * vae.vae_scaling_factor_image
    return latent


def encode_text(prompt, tokenizer, text_encoder, max_text_seq_length, device):
    """Encode text following lora_trainer encode_text method."""
    prompt_token_ids = tokenizer(
        prompt,
        padding="max_length",
        max_length=max_text_seq_length,
        truncation=True,
        add_special_tokens=True,
        return_tensors="pt",
    )
    prompt_token_ids = prompt_token_ids.input_ids
    prompt_embedding = text_encoder(prompt_token_ids.to(device))[0]
    return prompt_embedding


def main(args):
    data_dir = Path(args.out)
    os.makedirs(data_dir, exist_ok=True)
    os.makedirs(data_dir / "videos", exist_ok=True)
    os.makedirs(data_dir / "video_latents", exist_ok=True)
    os.makedirs(data_dir / "pointmap", exist_ok=True)
    os.makedirs(data_dir / "pointmap_latents", exist_ok=True)
    os.makedirs(data_dir / "datalist", exist_ok=True)
    os.makedirs(data_dir / "prompts", exist_ok=True)
    os.makedirs(data_dir / "center", exist_ok=True)
    os.makedirs(data_dir / "scale", exist_ok=True)
    
    max_frames = 81     # CogVideo 1.5 required framelength
    resolution_out = (args.resolution_h, args.resolution_w)     # CogVideo 1.5 required resolution
    resolution_crop = resolution_out

    # Create cache directories following i2pm_dataset.py structure
    cache_dir = data_dir / "cache_new"
    train_resolution_str = f"{max_frames}x{args.resolution_h}x{args.resolution_w}"
    video_latent_dir = cache_dir / "video_latent" / "cogvideox-i2dpm" / train_resolution_str
    prompt_embeddings_dir = cache_dir / "prompt_embeddings"
    video_latent_dir.mkdir(parents=True, exist_ok=True)
    prompt_embeddings_dir.mkdir(parents=True, exist_ok=True)

    # Load dataset samples
    pexel_datalist = []
    with open('data/filtered_pexelx.txt', 'r') as f:
        for line in f.readlines():
            pexel_datalist.append(line.strip())
    pexel_datalist_segment = split_list(pexel_datalist, num_segments=args.num_tasks)[args.task_idx]
    print(f'{args.task_idx}/{args.num_tasks} task | {len(pexel_datalist_segment)}/{len(pexel_datalist)} samples')
    
    # Initialize dataset
    cache_dir_tmp = f'.cache/{args.task_idx:05d}/'
    train_dataset_iterator = iter(
            PointmapDataset(
                datalist = pexel_datalist_segment,
                max_frames = max_frames,
                s3_conf_path = '~/petreloss.conf',
                debug=False,
                random_shuffle=False,
                cache_dir=cache_dir_tmp,
                skip_invalid=False
        )
    )

    # Initialize tokenizers and models
    # 1. VAE for video encoding
    vae_model_path = 'pretrained/CogVideoX1.5-5B-I2V/vae/'
    vae = CogvideoTokenizer(model_path=vae_model_path)
    
    # 2. Text encoder and tokenizer for prompt encoding
    model_path = 'pretrained/CogVideoX1.5-5B-I2V/'
    text_tokenizer = AutoTokenizer.from_pretrained(model_path, subfolder="tokenizer")
    text_encoder = T5EncoderModel.from_pretrained(model_path, subfolder="text_encoder")
    text_encoder.to(args.device)
    
    # Get transformer config to match lora_trainer.py
    transformer_config = {
        "max_text_seq_length": 226  # Default value, adjust if needed
    }

    prompt_list = []
    data_list = []
    center_list = []
    scale_list = []
    
    for i in range(len(pexel_datalist_segment)):
        data = next(train_dataset_iterator)
        if data is None:
            continue
            
        # Get caption and prepare prompt
        caption = data.video.caption
        prompt_suffix = 'POINTMAP_STYLE.'
        full_prompt = caption + ' ' + prompt_suffix
        prompt_list.append(caption)
        
        # Generate video name as in i2pm_dataset.py
        video_name = os.path.basename(data.video.path).replace('.mp4', '')
        if data.clip_start > -1:
            video_name = video_name + f'-{data.clip_start:04d}-{data.clip_start+data.length:04d}'
        
        # Define file paths
        video_path = data_dir / "videos" / f"{video_name}.mp4"
        pm_path = data_dir / "pointmap" / f"{video_name}.mp4"
        pointmap_latent_path = data_dir / "pointmap_latents" / f"{video_name}.pt"
        data_list.append(f'videos/{video_name}.mp4')
        
        # Cache prompt embedding (following i2pm_dataset.py)
        prompt_hash = str(hashlib.sha256(full_prompt.encode()).hexdigest())
        prompt_embedding_path = prompt_embeddings_dir / (prompt_hash + ".safetensors")
        
        if not prompt_embedding_path.exists():
            prompt_embedding = encode_text(
                full_prompt, 
                text_tokenizer, 
                text_encoder, 
                transformer_config["max_text_seq_length"],
                args.device
            )
            # Convert to CPU and format as in i2pm_dataset.py
            prompt_embedding = prompt_embedding.to("cpu")
            # [1, seq_len, hidden_size] -> [seq_len, hidden_size]
            prompt_embedding = prompt_embedding[0]
            save_file({"prompt_embedding": prompt_embedding}, prompt_embedding_path)
            print(f"Saved prompt embedding to {prompt_embedding_path}")
        
        # Process RGB video if it doesn't exist
        if not video_path.exists():
            rgb = data.rgb_raw
            # Get current dimensions and target dimensions
            T, H, W, C = rgb.shape
            target_h, target_w = resolution_out
            
            # Calculate resize dimensions maintaining aspect ratio
            if H/W < target_h/target_w:  # Height is shorter side
                new_h = target_h
                new_w = int(W * (target_h/H))
            else:  # Width is shorter side
                new_w = target_w 
                new_h = int(H * (target_w/W))
                
            # Resize rgb to match shortest side
            rgb = nnf.interpolate(torch.from_numpy(rgb.transpose(0,3,1,2)), 
                                size=(new_h, new_w), 
                                mode='bilinear', 
                                align_corners=False,
                                antialias=True).numpy().transpose(0,2,3,1)
            rgb = F.center_crop(torch.from_numpy(rgb.transpose(0,3,1,2)), resolution_out).numpy().transpose(0,2,3,1)

            # Extend video length to max_frames if needed
            if rgb.shape[0] < max_frames:
                num_original = T
                num_needed = max_frames - num_original
                reverse_rgb = rgb[::-1]
                rgb = np.concatenate([
                    rgb,  # Original sequence
                    reverse_rgb[:num_needed]  # Remaining frames needed
                ], axis=0)
            
            # Save the processed RGB video
            imageio.mimwrite(video_path, (rgb * 255).clip(0, 255).astype(np.uint8), fps=24)
            print(f"Saved RGB video to {video_path}")
        else:
            print(f"RGB video already exists at {video_path}, skipping processing")
            # Load the video to get rgb for encoding
            rgb = []  # Will be set below if needed
        
        # Cache encoded video
        encoded_video_path = video_latent_dir / (video_name + ".safetensors")
        if not encoded_video_path.exists():
            # If we skipped processing the video, load it now
            if not rgb:
                import decord
                decord.bridge.set_bridge("torch")
                video_reader = decord.VideoReader(uri=str(video_path))
                rgb = video_reader[:].float() / 255.0
            else:
                # Prepare rgb for encoding if we processed it above
                raise NotImplementedError("Not implemented")
                rgb_tensor = torch.from_numpy(rgb.transpose(0, 3, 1, 2)).float()
                rgb_tensor = rgb_tensor / 255.0 * 2.0 - 1.0  # Normalize to [-1, 1]
                rgb_tensor = rgb_tensor.unsqueeze(0)  # Add batch dimension [1, F, C, H, W]
                rgb = rgb_tensor
            
            # Encode video
            encoded_video = encode_video(rgb, vae)
            # [1, C, F, H, W] -> [C, F, H, W]
            encoded_video = encoded_video[0].cpu()
            save_file({"encoded_video": encoded_video}, encoded_video_path)
            print(f"Saved encoded video to {encoded_video_path}")
        
        # Process pointmap if it doesn't exist
        if not pm_path.exists() or not pointmap_latent_path.exists():
            pointmap = data.pointmap.pcd
            pointmap = np.clip(pointmap, np.percentile(pointmap, 2, axis=1, keepdims=True), 
                               np.percentile(pointmap, 98, axis=1, keepdims=True))
            pointmap_in, center, scale = vae.normalize_instance(pointmap, range_min=0)
            center_list.append(center)
            scale_list.append(scale)
            
            # Reshape and process pointmap
            pointmap_in = pointmap_in.reshape(*data.pointmap.rgb.shape)
            
            # Get dimensions for resizing
            T, H, W, C = data.rgb_raw.shape
            target_h, target_w = resolution_out
            
            # Calculate resize dimensions maintaining aspect ratio
            if H/W < target_h/target_w:
                new_h = target_h
                new_w = int(W * (target_h/H))
            else:
                new_w = target_w 
                new_h = int(H * (target_w/W))
            
            # Resize and crop
            pointmap_in = nnf.interpolate(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), 
                                size=(new_h, new_w), 
                                mode='nearest').numpy().transpose(0,2,3,1)
            pointmap_in = F.center_crop(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), 
                                        resolution_crop).numpy().transpose(0,2,3,1)
            
            # Extend pointmap length to max_frames if needed
            if pointmap_in.shape[0] < max_frames:
                # Calculate frames needed
                num_needed = max_frames - pointmap_in.shape[0]
                # Create reversed sequence for padding
                reverse_pointmap_in = pointmap_in[::-1]
                # Concatenate original with reversed frames
                pointmap_in = np.concatenate([
                    pointmap_in,
                    reverse_pointmap_in[:num_needed]
                ], axis=0)
            
            # Save pointmap video and latents
            imageio.mimwrite(pm_path, (pointmap_in * 255).clip(0, 255).astype(np.uint8), fps=24)
            torch.save(vae.encode(pointmap_in)[0], pointmap_latent_path)
            print(f"Saved pointmap video to {pm_path} and latents to {pointmap_latent_path}")
        else:
            print(f"Pointmap already exists at {pm_path}, skipping processing")
            # If we still need the center and scale, load them from existing files
            if not center_list:
                # Try to load from existing files or use default values
                try:
                    center_path = data_dir / "center" / f"{args.task_idx:05d}.npy"
                    scale_path = data_dir / "scale" / f"{args.task_idx:05d}.npy"
                    if center_path.exists() and scale_path.exists():
                        centers = np.load(center_path)
                        scales = np.load(scale_path)
                        center_list = centers.tolist()
                        scale_list = scales.tolist()
                except Exception as e:
                    print(f"Could not load center and scale: {e}")
        
        print(f'Finished processing sample {i+1}/{len(pexel_datalist_segment)}.')

    # Save metadata
    with open(data_dir / "prompts" / f"{args.task_idx:05d}.txt", 'w') as f:
        f.write('\n'.join(prompt_list))
    
    with open(data_dir / "datalist" / f"{args.task_idx:05d}.txt", 'w') as f:
        f.write('\n'.join(data_list))

    # Save center and scale if we have them
    if center_list and scale_list:
        center_path = data_dir / "center" / f"{args.task_idx:05d}.npy"
        scale_path = data_dir / "scale" / f"{args.task_idx:05d}.npy"
        np.save(center_path, np.stack(center_list))
        np.save(scale_path, np.stack(scale_list))

    print(f'All finished.')

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='build pointmap dataset')
    parser.add_argument('--out', type=str, default='data/feb23_debug')
    parser.add_argument('--num_tasks', type=int, default=1000)
    parser.add_argument('--task_idx', type=int, default=0)
    parser.add_argument('--resolution_h', type=int, default=768)
    parser.add_argument('--resolution_w', type=int, default=1360)
    parser.add_argument('--device', type=str, default='cuda' if torch.cuda.is_available() else 'cpu')
    args = parser.parse_args()
    
    main(args)
    
