import os
import hashlib
from pathlib import Path
import numpy as np
import imageio
import torch
import torchvision.transforms.functional as F
import torch.nn.functional as nnf
import argparse
from core.dataset import PointmapDataset
from core.tokenizer.cogvideo import CogvideoTokenizer
from transformers import AutoTokenizer, T5EncoderModel
from safetensors.torch import save_file
from accelerate.logging import get_logger


LOG_NAME = "cogvideox-model"
LOG_LEVEL = "INFO"
logger = get_logger(LOG_NAME, LOG_LEVEL)


def split_list(datalist, num_segments):
    """Splits list datalist into num_segments continuous segments with balanced load."""
    n = len(datalist)
    segment_size = n // num_segments
    remainder = n % num_segments  # Extra elements to distribute

    segments = []
    start = 0
    for i in range(num_segments):
        extra = 1 if i < remainder else 0  # Distribute remainder among the first few segments
        end = start + segment_size + extra
        segments.append(datalist[start:end])
        start = end

    return segments


def main(args):
    data_dir = args.out
    os.makedirs(data_dir, exist_ok=True)
    os.makedirs(f'{data_dir}/videos', exist_ok=True)
    os.makedirs(f'{data_dir}/video_latents', exist_ok=True)
    os.makedirs(f'{data_dir}/pointmap', exist_ok=True)
    os.makedirs(f'{data_dir}/pointmap_latents', exist_ok=True)
    os.makedirs(f'{data_dir}/datalist', exist_ok=True)
    os.makedirs(f'{data_dir}/prompts', exist_ok=True)
    os.makedirs(f'{data_dir}/center', exist_ok=True)
    os.makedirs(f'{data_dir}/scale', exist_ok=True)
    
    # Create cache directories for encoded videos and prompt embeddings
    cache_dir = Path(data_dir) / "cache"
    train_resolution_str = f"{args.resolution_h}x{args.resolution_w}"
    video_latent_dir = cache_dir / "video_latent" / "cogvideox-i2dpm" / train_resolution_str
    prompt_embeddings_dir = cache_dir / "prompt_embeddings"
    video_latent_dir.mkdir(parents=True, exist_ok=True)
    prompt_embeddings_dir.mkdir(parents=True, exist_ok=True)

    max_frames = 81     # CogVideo 1.5 required framelength
    resolution_out = (args.resolution_h, args.resolution_w)     # CogVideo 1.5 required resolution
    resolution_crop = resolution_out

    pexel_datalist = []
    with open('data/filtered_pexelx.txt', 'r') as f:
        for line in f.readlines():
            pexel_datalist.append(line.strip())
    pexel_datalist_segment = split_list(pexel_datalist, num_segments=args.num_tasks)[args.task_idx]
    print(f'{args.task_idx}/{args.num_tasks} task | {len(pexel_datalist_segment)}/{len(pexel_datalist)} samples')
    # dataset
    cache_dir = f'.cache/{args.task_idx:05d}/'
    train_dataset_iterator = iter(
            PointmapDataset(
                datalist = pexel_datalist_segment,
                max_frames = max_frames,
                s3_conf_path = '~/petreloss.conf',
                debug=False,
                random_shuffle=False,
                cache_dir=cache_dir,
                skip_invalid=False
        )
    )

    # tokenizer for VAE encoding
    vae_model_path = 'pretrained/CogVideoX1.5-5B-I2V/vae/'
    # we do not apply scaling_factor when caching the latents of pointmap
    vae_tokenizer = CogvideoTokenizer(model_path=vae_model_path)
    
    # Load text encoder for prompt embeddings
    model_path = 'pretrained/CogVideoX1.5-5B-I2V/'
    text_tokenizer = AutoTokenizer.from_pretrained(model_path, subfolder="tokenizer")
    text_encoder = T5EncoderModel.from_pretrained(model_path, subfolder="text_encoder")
    text_encoder.to(args.device)
    
    # Function to encode text prompts
    def encode_text(prompt):
        text_inputs = text_tokenizer(
            prompt,
            padding="max_length",
            max_length=text_tokenizer.model_max_length,
            truncation=True,
            return_tensors="pt",
        )
        text_input_ids = text_inputs.input_ids.to(args.device)
        
        prompt_embeds = text_encoder(
            text_input_ids,
            attention_mask=text_inputs.attention_mask.to(args.device),
        )
        prompt_embeds = prompt_embeds[0]
        return prompt_embeds
    
    # Function to encode videos
    def encode_video(video):
        # Normalize video to [-1, 1]
        video = video.transpose(0, 3, 1, 2)  # [T, H, W, C] -> [T, C, H, W]
        video = torch.tensor(video).to(args.device) / 255.0 * 2.0 - 1.0
        # Convert to [B, C, F, H, W]
        video = video.unsqueeze(0).permute(0, 2, 1, 3, 4).contiguous()
        with torch.no_grad():
            # Get latent from VAE
            latent_dist = vae_tokenizer.vae.encode(video.to(dtype=vae_tokenizer.vae.dtype))
            latent = latent_dist.latent_dist.sample() * vae_tokenizer.vae.config.scaling_factor
        return latent

    prompt_list = []
    data_list = []
    center_list = []
    scale_list = []
    for i in range(len(pexel_datalist_segment)):
        data = next(train_dataset_iterator)
        if data is None:
            continue
        pm_resolution = data.pointmap.get_resolution()
        pm_num_frames = data.pointmap.num_frames()

        caption = data.video.caption
        # Cache prompt embedding
        prompt_suffix = 'POINTMAP_STYLE.'
        full_prompt = caption + ' ' + prompt_suffix
        prompt_list.append(caption)
        prompt_hash = str(hashlib.sha256(full_prompt.encode()).hexdigest())
        prompt_embedding_path = prompt_embeddings_dir / (prompt_hash + ".safetensors")
        
        if not prompt_embedding_path.exists():
            prompt_embedding = encode_text(full_prompt)
            prompt_embedding = prompt_embedding.to("cpu")
            # [1, seq_len, hidden_size] -> [seq_len, hidden_size]
            prompt_embedding = prompt_embedding[0]
            save_file({"prompt_embedding": prompt_embedding}, prompt_embedding_path)
            logger.info(f"Saved prompt embedding to {prompt_embedding_path}")
        
        video_name = os.path.basename(data.video.path).replace('.mp4', '')
        if data.clip_start > -1:
            video_name = video_name + f'-{data.clip_start:04d}-{data.clip_start+data.length:04d}'
        video_path = f'{data_dir}/videos/{video_name}.mp4'
        data_list.append(f'videos/{video_name}.mp4')
        
        # save rgb
        rgb = data.rgb_raw
        # Get current dimensions and target dimensions
        T, H, W, C = rgb.shape
        target_h, target_w = resolution_out
        
        # Calculate resize dimensions to match shortest side while maintaining aspect ratio
        if H/W < target_h/target_w:  # Height is shorter side
            new_h = target_h
            new_w = int(W * (target_h/H))
        else:  # Width is shorter side
            new_w = target_w 
            new_h = int(H * (target_w/W))
            
        # Resize rgb to match shortest side
        rgb = nnf.interpolate(torch.from_numpy(rgb.transpose(0,3,1,2)), 
                            size=(new_h, new_w), 
                            mode='bilinear', 
                            align_corners=False,
                            antialias=True).numpy().transpose(0,2,3,1)
        rgb = F.center_crop(torch.from_numpy(rgb.transpose(0,3,1,2)), resolution_out).numpy().transpose(0,2,3,1)

        # Extend video length to max_frames by reverse padding if needed
        if rgb.shape[0] < max_frames:
            # Calculate number of frames needed
            num_original = T
            num_needed = max_frames - num_original
            
            # Create reversed sequence
            reverse_rgb = rgb[::-1]
            
            # Concatenate original frames with repeated reverse frames
            rgb = np.concatenate([
                rgb,  # Original sequence
                reverse_rgb[:num_needed]  # Remaining frames needed
            ], axis=0)
        imageio.mimwrite(video_path, (rgb * 255).clip(0, 255).astype(np.uint8), fps=24)
        
        # Cache encoded video
        encoded_video_path = video_latent_dir / (video_name + ".safetensors")
        if not encoded_video_path.exists():
            encoded_video = encode_video(rgb)
            # [1, C, F, H, W] -> [C, F, H, W]
            encoded_video = encoded_video[0].cpu()
            save_file({"encoded_video": encoded_video}, encoded_video_path)
            logger.info(f"Saved encoded video to {encoded_video_path}")
            
        # save pointmap latents
        pointmap_latent_path = f'{data_dir}/pointmap_latents/{video_name}.pt'
        pointmap = data.pointmap.pcd
        pointmap = np.clip(pointmap, np.percentile(pointmap, 2, axis=1, keepdims=True), np.percentile(pointmap, 98,  axis=1, keepdims=True))
        pointmap_in, center, scale = vae_tokenizer.normalize_instance(pointmap, range_min=0)
        center_list.append(center)
        scale_list.append(scale)
        pointmap_in = pointmap_in.reshape(*data.pointmap.rgb.shape)
        pointmap_in = nnf.interpolate(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), 
                            size=(new_h, new_w), 
                            mode='nearest').numpy().transpose(0,2,3,1)
        pointmap_in = F.center_crop(torch.from_numpy(pointmap_in.transpose(0,3,1,2)), resolution_crop).numpy().transpose(0,2,3,1)
        if pointmap_in.shape[0] < max_frames:
            # Create reversed sequence
            reverse_pointmap_in = pointmap_in[::-1]
            
            # Concatenate original frames with repeated reverse frames
            pointmap_in = np.concatenate([
                pointmap_in,  # Original sequence
                reverse_pointmap_in[:num_needed]  # Remaining frames needed
            ], axis=0)

        pm_path = f'{data_dir}/pointmap/{video_name}.mp4'
        imageio.mimwrite(pm_path, (pointmap_in * 255).clip(0, 255).astype(np.uint8), fps=24)
        torch.save(vae_tokenizer.encode(pointmap_in)[0], pointmap_latent_path)
        print(f'Finish {i}.')

    # backup segment meta data
    with open(f'{data_dir}/prompts/{args.task_idx:05d}.txt', 'w') as f:
        f.write('\n'.join(prompt_list))
    with open(f'{data_dir}/datalist/{args.task_idx:05d}.txt', 'w') as f:
        f.write('\n'.join(data_list))

    center_path = f'{data_dir}/center/{args.task_idx:05d}.npy'
    scale_path = f'{data_dir}/scale/{args.task_idx:05d}.npy'
    np.save(center_path, np.stack(center_list))
    np.save(scale_path, np.stack(scale_list))

    print(f'All finished.')

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='build pointmap dataset')
    parser.add_argument('--out', type=str, default='data/feb23_debug')
    parser.add_argument('--num_tasks', type=int, default=1000)
    parser.add_argument('--task_idx', type=int, default=0)
    parser.add_argument('--resolution_h', type=int, default=768)
    parser.add_argument('--resolution_w', type=int, default=1360)
    parser.add_argument('--device', type=str, default='cuda' if torch.cuda.is_available() else 'cpu')
    args = parser.parse_args()
    
    main(args)
    
