import sys
from pathlib import Path
import os
import torch
import numpy as np
import imageio
from tqdm import tqdm

sys.path.append(str(Path(__file__).parent.parent))

from core.finetune.schemas import Args
from core.finetune.models.wan_i2v.sft_trainer import WanI2VSftTrainer
from core.tokenizer.wan import WanTokenizer

# ---- Configuration ----
model_path = Path('pretrained/Wan2.1-I2V-14B-480P-Diffusers')
output_dir = Path('results/dataloader_visualize')
os.makedirs(output_dir, exist_ok=True)

# Fill these according to your dataset/cache
args = Args(
    model_path=model_path,
    model_name='wan-i2v',
    model_type='wan-i2v',
    training_type='sft',
    output_dir=output_dir,
    data_root=Path('data/wan21'),
    caption_column=Path('captions.txt'),
    image_column=None,
    video_column=Path('prompts.txt'),
    train_epochs=1,
    train_steps=1,
    batch_size=1,
    gradient_accumulation_steps=1,
    train_resolution=(49, 480, 720),
    mixed_precision='bf16',
    report_to='tensorboard',
    num_workers=0,
    pin_memory=False,
    seed=42,
    validation_steps=None,
    validation_dir=None,
    validation_prompts=None,
    validation_images=None,
    validation_videos=None,
)

# ---- Trainer and Tokenizer ----
trainer = WanI2VSftTrainer(args)
trainer.prepare_models()
trainer.prepare_dataset()

tokenizer = WanTokenizer(model_path=str(model_path / 'vae'))

def save_video(frames, path):
    frames = np.clip(frames * 255, 0, 255).astype(np.uint8)
    imageio.mimwrite(path, frames, fps=8)

def save_image(img, path):
    img = np.clip((img * 255), 0, 255).astype(np.uint8)
    imageio.imwrite(path, img.transpose(1, 2, 0))

# ---- Iterate and Visualize ----
for i, batch in enumerate(tqdm(trainer.data_loader)):
    # Save the first image in the batch
    img = batch['images'][0].cpu().numpy()  # [C, H, W]
    save_image(img, output_dir / f'batch_{i}_image.png')

    # Decode the video latent
    latent = batch['encoded_videos'][0].unsqueeze(0).to(tokenizer.device, tokenizer.dtype)  # [1, C, F, H, W]
    video = tokenizer.decode(latent)  # [F, H, W, C]
    save_video(video, output_dir / f'batch_{i}_video.mp4')
    # Also save first frame as PNG
    save_image(video[0].transpose(2, 0, 1), output_dir / f'batch_{i}_video_first_frame.png')

    if i >= 9:
        break  # Only visualize first 10 batches for brevity
