import os
import cv2
import numpy as np
import torch
from tqdm import tqdm
from typing import List, Tuple, Optional
import argparse
from pathlib import Path
import json
import random
from PIL import Image
import torchvision.transforms as T
from torchvision.transforms.functional import resize, InterpolationMode

def get_video_info(video_path: str) -> Tuple[int, int, int, int]:
    """Get video information including width, height, fps, and total frames."""
    cap = cv2.VideoCapture(video_path)
    if not cap.isOpened():
        raise ValueError(f"Could not open video: {video_path}")
    
    width = int(cap.get(cv2.CAP_PROP_FRAME_WIDTH))
    height = int(cap.get(cv2.CAP_PROP_FRAME_HEIGHT))
    fps = int(cap.get(cv2.CAP_PROP_FPS))
    total_frames = int(cap.get(cv2.CAP_PROP_FRAME_COUNT))
    
    cap.release()
    return width, height, fps, total_frames

def resize_frame(frame: np.ndarray, target_size: Tuple[int, int], is_pointmap: bool = False) -> np.ndarray:
    """Resize frame to target size, handling both RGB and pointmap data."""
    if is_pointmap:
        # For pointmaps, we need to preserve the depth values
        # First normalize to [0, 1]
        frame = frame.astype(np.float32)
        min_val = frame.min()
        max_val = frame.max()
        if max_val > min_val:
            frame = (frame - min_val) / (max_val - min_val)
        
        # Resize using nearest neighbor to preserve depth values
        frame = cv2.resize(frame, target_size, interpolation=cv2.INTER_NEAREST)
        
        # Scale back to original range
        if max_val > min_val:
            frame = frame * (max_val - min_val) + min_val
        return frame
    else:
        # For RGB, use bilinear interpolation
        return cv2.resize(frame, target_size, interpolation=cv2.INTER_LINEAR)

def process_video(
    video_path: str,
    output_path: str,
    target_frames: int = 81,
    target_size: Tuple[int, int] = (1360, 768),
    is_pointmap: bool = False,
    start_frame: Optional[int] = None,
    end_frame: Optional[int] = None,
) -> None:
    """Process a video to match target frames and size."""
    # Create output directory if it doesn't exist
    os.makedirs(output_path, exist_ok=True)
    
    # Get video information
    width, height, fps, total_frames = get_video_info(video_path)
    
    # Calculate frame indices to sample
    if start_frame is None:
        start_frame = 0
    if end_frame is None:
        end_frame = total_frames
    
    # Calculate frame indices to sample
    frame_indices = np.linspace(start_frame, end_frame - 1, target_frames, dtype=int)
    
    # Open video
    cap = cv2.VideoCapture(video_path)
    if not cap.isOpened():
        raise ValueError(f"Could not open video: {video_path}")
    
    # Process each frame
    for i, frame_idx in enumerate(tqdm(frame_indices, desc="Processing frames")):
        # Set frame position
        cap.set(cv2.CAP_PROP_POS_FRAMES, frame_idx)
        ret, frame = cap.read()
        if not ret:
            raise ValueError(f"Could not read frame {frame_idx} from video: {video_path}")
        
        # Convert to RGB if needed
        if not is_pointmap:
            frame = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
        
        # Resize frame
        frame = resize_frame(frame, target_size, is_pointmap)
        
        # Save frame
        frame_path = os.path.join(output_path, f"frame_{i:04d}.png")
        if is_pointmap:
            # For pointmaps, save as 16-bit PNG to preserve depth information
            cv2.imwrite(frame_path, frame.astype(np.uint16))
        else:
            # For RGB, save as 8-bit PNG
            cv2.imwrite(frame_path, cv2.cvtColor(frame, cv2.COLOR_RGB2BGR))
    
    cap.release()

def process_directory(
    input_dir: str,
    output_dir: str,
    target_frames: int = 81,
    target_size: Tuple[int, int] = (1360, 768),
    is_pointmap: bool = False,
    start_frame: Optional[int] = None,
    end_frame: Optional[int] = None,
) -> None:
    """Process all videos in a directory."""
    # Get all video files
    video_extensions = ['.mp4', '.avi', '.mov']
    video_files = []
    for ext in video_extensions:
        video_files.extend(list(Path(input_dir).glob(f"**/*{ext}")))
    
    # Process each video
    for video_path in tqdm(video_files, desc="Processing videos"):
        # Create output path
        rel_path = video_path.relative_to(input_dir)
        output_path = Path(output_dir) / rel_path.parent / video_path.stem
        
        try:
            process_video(
                str(video_path),
                str(output_path),
                target_frames,
                target_size,
                is_pointmap,
                start_frame,
                end_frame,
            )
        except Exception as e:
            print(f"Error processing {video_path}: {e}")

def main():
    parser = argparse.ArgumentParser(description="Process videos to match target frames and size")
    parser.add_argument("--input_dir", type=str, required=True, help="Input directory containing videos")
    parser.add_argument("--output_dir", type=str, required=True, help="Output directory for processed videos")
    parser.add_argument("--target_frames", type=int, default=81, help="Target number of frames")
    parser.add_argument("--target_width", type=int, default=1360, help="Target width")
    parser.add_argument("--target_height", type=int, default=768, help="Target height")
    parser.add_argument("--is_pointmap", action="store_true", help="Process as pointmap data")
    parser.add_argument("--start_frame", type=int, help="Start frame index")
    parser.add_argument("--end_frame", type=int, help="End frame index")
    
    args = parser.parse_args()
    
    process_directory(
        args.input_dir,
        args.output_dir,
        args.target_frames,
        (args.target_width, args.target_height),
        args.is_pointmap,
        args.start_frame,
        args.end_frame,
    )

if __name__ == "__main__":
    main() 