import { _optionalChain } from '@sentry/utils';
import { KoaInstrumentation } from '@opentelemetry/instrumentation-koa';
import { ATTR_HTTP_ROUTE } from '@opentelemetry/semantic-conventions';
import { defineIntegration, getIsolationScope, getDefaultIsolationScope, spanToJSON, captureException, SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN, SEMANTIC_ATTRIBUTE_SENTRY_OP } from '@sentry/core';
import { logger } from '@sentry/utils';
import { DEBUG_BUILD } from '../../debug-build.js';
import { generateInstrumentOnce } from '../../otel/instrument.js';
import { ensureIsWrapped } from '../../utils/ensureIsWrapped.js';

const INTEGRATION_NAME = 'Koa';

const instrumentKoa = generateInstrumentOnce(
  INTEGRATION_NAME,
  () =>
    new KoaInstrumentation({
      requestHook(span, info) {
        addKoaSpanAttributes(span);

        if (getIsolationScope() === getDefaultIsolationScope()) {
          DEBUG_BUILD && logger.warn('Isolation scope is default isolation scope - skipping setting transactionName');
          return;
        }
        const attributes = spanToJSON(span).data;
        const route = attributes && attributes[ATTR_HTTP_ROUTE];
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        const method = _optionalChain([info, 'optionalAccess', _ => _.context, 'optionalAccess', _2 => _2.request, 'optionalAccess', _3 => _3.method, 'optionalAccess', _4 => _4.toUpperCase, 'call', _5 => _5()]) || 'GET';
        if (route) {
          getIsolationScope().setTransactionName(`${method} ${route}`);
        }
      },
    }),
);

const _koaIntegration = (() => {
  return {
    name: INTEGRATION_NAME,
    setupOnce() {
      instrumentKoa();
    },
  };
}) ;

const koaIntegration = defineIntegration(_koaIntegration);

// eslint-disable-next-line @typescript-eslint/no-explicit-any
const setupKoaErrorHandler = (app) => {
  app.use(async (ctx, next) => {
    try {
      await next();
    } catch (error) {
      captureException(error);
      throw error;
    }
  });

  ensureIsWrapped(app.use, 'koa');
};

function addKoaSpanAttributes(span) {
  span.setAttribute(SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN, 'auto.http.otel.koa');

  const attributes = spanToJSON(span).data || {};

  // this is one of: middleware, router
  const type = attributes['koa.type'];

  if (type) {
    span.setAttribute(SEMANTIC_ATTRIBUTE_SENTRY_OP, `${type}.koa`);
  }

  // Also update the name
  const name = attributes['koa.name'];
  if (typeof name === 'string') {
    // Somehow, name is sometimes `''` for middleware spans
    // See: https://github.com/open-telemetry/opentelemetry-js-contrib/issues/2220
    span.updateName(name || '< unknown >');
  }
}

export { instrumentKoa, koaIntegration, setupKoaErrorHandler };
